<?php
// qbo_sales_revenue_report.php
// REFACTORED VERSION: Uses the local database as the "master" for organization names and fuzzy-matches QBO data to it.

// --- Essential Includes & Setup ---
require_once __DIR__ . '/dbconn.php';
require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/qbo_functions.php';
require_once __DIR__ . '/config/qbo_config.php';
global $conn, $qboBaseConfig;
require_once __DIR__ . '/qbo_bootstrap.php';
global $qboUtil;

// --- Filter Logic ---
$filters = [
    'date_from' => $_GET['date_from'] ?? date('Y-m-d', strtotime('-90 days')),
    'date_to' => $_GET['date_to'] ?? date('Y-m-d'),
    'region' => isset($_GET['region']) && is_array($_GET['region']) ? $_GET['region'] : [],
    'country' => $_GET['country'] ?? 'All'
];

// --- Data Fetching Logic ---
$final_report_data = [];
$error_message = null;

try {
    // --- Step 1: Fetch LOCAL organizations based on filters to build the MASTER search map ---
    $local_org_data = [];
    $sql_local_orgs = "SELECT organizationid, organization_name, region, country FROM tdu_organisation WHERE organization_name IS NOT NULL AND organization_name != ''";
    $sql_local_orgs_params = [];
    $sql_local_orgs_types = "";
    
    // Add Region filter
    if (!empty($filters['region'])) {
        $region_placeholders = implode(',', array_fill(0, count($filters['region']), '?'));
        $sql_local_orgs .= " AND region IN ({$region_placeholders})";
        foreach ($filters['region'] as $region) {
            $sql_local_orgs_params[] = $region;
            $sql_local_orgs_types .= "s";
        }
    }
    
    // Add Country filter
    if ($filters['country'] !== 'All') {
        $sql_local_orgs .= " AND country = ?";
        $sql_local_orgs_params[] = $filters['country'];
        $sql_local_orgs_types .= "s";
    }
    
    $stmt_local_orgs = $conn->prepare($sql_local_orgs);
    
    if ($stmt_local_orgs) {
        if (!empty($sql_local_orgs_params)) {
            $stmt_local_orgs->bind_param($sql_local_orgs_types, ...$sql_local_orgs_params);
        }
        $stmt_local_orgs->execute();
        $result_local_orgs = $stmt_local_orgs->get_result();
    
        while ($row = $result_local_orgs->fetch_assoc()) {
            $local_org_data[$row['organization_name']] = $row; // Key by name for easy access later
        }
        $stmt_local_orgs->close();
    } else {
         throw new Exception("Failed to prepare SQL statement for local organizations: " . $conn->error);
    }
    
    // Create an array of objects suitable for the search map builder
    $org_objects_for_map = array_map(function($name) { return (object)['DisplayName' => $name]; }, array_keys($local_org_data));
    $localOrgSearchMap = $qboUtil->buildReportSearchMap($org_objects_for_map);

    // --- Step 2: Fetch all QBO transactions (Invoices) ---
    $qbo_invoice_map = [];
    if ($qboUtil) {
        $customer_ids_from_txns = [];
        $all_qbo_invoices = [];
        $startPosition = 1;
        while (true) {
            $invoice_query = "SELECT CustomerRef, TotalAmt FROM Invoice WHERE TxnDate >= '{$filters['date_from']}' AND TxnDate <= '{$filters['date_to']}' STARTPOSITION {$startPosition} MAXRESULTS 1000";
            $invoicesBatch = $qboUtil->getDataService()->Query($invoice_query);
            if (empty($invoicesBatch)) break;
            foreach ($invoicesBatch as $invoice) {
                $all_qbo_invoices[] = $invoice;
                if (isset($invoice->CustomerRef)) $customer_ids_from_txns[] = $invoice->CustomerRef;
            }
            if (count($invoicesBatch) < 1000) break;
            $startPosition += 1000;
        }
        
        $customer_id_to_name_map = [];
        $unique_customer_ids = array_unique(array_filter($customer_ids_from_txns));
        if (!empty($unique_customer_ids)) {
            $id_list_string = "'" . implode("','", $unique_customer_ids) . "'";
            $customer_query = "SELECT Id, DisplayName FROM Customer WHERE Id IN ({$id_list_string})";
            $qbo_customers = $qboUtil->getDataService()->Query($customer_query);
            foreach ($qbo_customers as $customer) { $customer_id_to_name_map[$customer->Id] = $customer->DisplayName; }
        }
        
        foreach ($all_qbo_invoices as $invoice) {
            $customerName = $customer_id_to_name_map[$invoice->CustomerRef] ?? null;
            if ($customerName) {
                $qbo_invoice_map[$customerName] = ($qbo_invoice_map[$customerName] ?? 0) + (float)$invoice->TotalAmt;
            }
        }
    }

    // --- Step 3: Fetch Local Quote & Pax Data ---
    $pax_and_quote_map = [];
    $sql_params = []; $sql_types = "";
    $sql_quotes = "SELECT org.organization_name, (q.adults + q.children) AS pax FROM vtiger_quotes AS q JOIN vtiger_quotescf AS qcf ON q.quoteid = qcf.quoteid JOIN tdu_organisation AS org ON q.accountid = org.organizationid WHERE qcf.cf_1162 BETWEEN ? AND ?";
    $sql_params[] = $filters['date_from']; $sql_params[] = $filters['date_to']; $sql_types .= "ss";
    if (!empty($filters['region'])) { $region_placeholders = implode(',', array_fill(0, count($filters['region']), '?')); $sql_quotes .= " AND org.region IN ({$region_placeholders})"; foreach ($filters['region'] as $region) { $sql_params[] = $region; $sql_types .= "s"; } }
    if ($filters['country'] !== 'All') { $sql_quotes .= " AND org.country = ?"; $sql_params[] = $filters['country']; $sql_types .= "s"; }
    
    $stmt_quotes = $conn->prepare($sql_quotes);
    if ($stmt_quotes) {
        $stmt_quotes->bind_param($sql_types, ...$sql_params);
        $stmt_quotes->execute();
        $quote_data = $stmt_quotes->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt_quotes->close();
    }
    foreach($quote_data as $quote_row) {
        $orgName = $quote_row['organization_name'];
        if (!isset($pax_and_quote_map[$orgName])) {
            $pax_and_quote_map[$orgName] = ['pax' => 0, 'quote_count' => 0];
        }
        $pax_and_quote_map[$orgName]['pax'] += (int)$quote_row['pax'];
        $pax_and_quote_map[$orgName]['quote_count']++;
    }
    
    // --- Step 4: Aggregate all data, using the local database as the master list ---
    $aggregated_report_data = [];
    // Initialize the report, but ONLY with local organizations that have a quote count > 0
    foreach ($local_org_data as $orgName => $orgDetails) {
        $quoteCount = $pax_and_quote_map[$orgName]['quote_count'] ?? 0;
        // --- THIS IS THE NEW CONDITION ---
        if ($quoteCount > 0) {
            $aggregated_report_data[$orgName] = [
                'organizationid' => $orgDetails['organizationid'],
                'organization_name' => $orgName,
                'region' => $orgDetails['region'],
                'country' => $orgDetails['country'],
                'qbo_total_amount' => 0,
                'total_pax' => $pax_and_quote_map[$orgName]['pax'] ?? 0,
                'quote_count' => $quoteCount
            ];
        }
    }
    
    // Loop through QBO data and map it back to the local master list via fuzzy match
    foreach ($qbo_invoice_map as $qboCustomerName => $totalAmount) {
        $matchedLocalName = $qboUtil->findBestEntityMatch($qboCustomerName, $localOrgSearchMap);
        
        if ($matchedLocalName && isset($aggregated_report_data[$matchedLocalName])) {
            $aggregated_report_data[$matchedLocalName]['qbo_total_amount'] += $totalAmount;
        }
    }
    
    $final_report_data = array_values($aggregated_report_data);

} catch (Exception $e) {
    $error_message = "An error occurred: " . htmlspecialchars($e->getMessage());
}
?>

<!-- ========================================================================= -->
<!-- UI, LIBRARIES, TABLE, AND JAVASCRIPT (No changes needed below this line)  -->
<!-- ========================================================================= -->
<link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/1.13.6/css/jquery.dataTables.min.css">
<script type="text/javascript" charset="utf8" src="https://code.jquery.com/jquery-3.7.0.js"></script>
<script type="text/javascript" charset="utf8" src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>

<style>
    .multiselect-container { position: relative; display: inline-block; min-width: 200px; } .select-box { border: 1px solid #ccc; border-radius: 4px; padding: 8px; cursor: pointer; background: #fff; display: flex; justify-content: space-between; align-items: center; } .select-box .selected-options { overflow: hidden; white-space: nowrap; text-overflow: ellipsis; } .select-box::after { content: '▼'; font-size: 0.8em; } .checkboxes { display: none; position: absolute; background-color: #fff; border: 1px solid #ccc; border-radius: 4px; z-index: 100; max-height: 200px; overflow-y: auto; min-width: 100%; padding: 5px; } .checkboxes label { display: block; padding: 5px 10px; } .checkboxes label:hover { background-color: #f1f1f1; } .checkboxes label input { margin-right: 10px; }
    #salesReportTable_filter { margin-bottom: 10px; } tr.dt-hasChild td.dt-control { background-image: url('https://datatables.net/examples/resources/details_close.png'); } td.dt-control { background: url('https://datatables.net/examples/resources/details_open.png') no-repeat center center; cursor: pointer; } .detail-container { padding: 10px 15px; background-color: #fafafa; display: flex; gap: 30px;} .detail-section { flex: 1; } .detail-container h4 { margin-top: 0; margin-bottom: 10px; border-bottom: 1px solid #ddd; padding-bottom: 5px; } .detail-table { width: 100%; font-size: 0.9em; } .detail-table th, .detail-table td { border: none; border-bottom: 1px solid #e9ecef; padding: 6px 4px; } .status-paid { color: #2d8a32; } .status-open { color: #0077c5; } .status-overdue { color: #d93025; font-weight: bold; }
</style>

<div class="report-header"><h2>Sales & Revenue Report</h2></div>

<?php if ($error_message): ?>
    <div class='no-data-message' style='color:red; text-align:center; padding: 10px; border: 1px solid red; background: #ffebee;'><?= $error_message ?></div>
<?php endif; ?>

<form method="GET" class="filter-form">
    <input type="hidden" name="report" value="sales_revenue">
    <div><label for="date_from">From:</label><input type="date" id="date_from" name="date_from" value="<?= htmlspecialchars($filters['date_from']) ?>"></div>
    <div><label for="date_to">To:</label><input type="date" id="date_to" name="date_to" value="<?= htmlspecialchars($filters['date_to']) ?>"></div>
    <div>
        <label>Region:</label>
        <div class="multiselect-container">
            <div class="select-box" onclick="toggleCheckboxes('region-checkboxes')"><span class="selected-options" id="region-selected-options">Select Region(s)</span></div>
            <div class="checkboxes" id="region-checkboxes">
                <?php $region_sql = "SELECT DISTINCT region FROM tdu_organisation WHERE region IS NOT NULL AND region != '' ORDER BY region"; $region_result = mysqli_query($conn, $region_sql); while($row = mysqli_fetch_assoc($region_result)) { $checked = in_array($row['region'], $filters['region']) ? 'checked' : ''; echo "<label><input type='checkbox' name='region[]' value='" . htmlspecialchars($row['region']) . "' {$checked} onchange=\"updateSelectedOptions('region')\"/>" . htmlspecialchars($row['region']) . "</label>"; } ?>
            </div>
        </div>
    </div>
    <div>
        <label for="country">Country:</label>
        <select id="country" name="country">
            <option value="All">All Countries</option>
            <option value="Australia" <?= $filters['country'] === 'Australia' ? 'selected' : '' ?>>Australia</option>
            <option value="New Zealand" <?= $filters['country'] === 'New Zealand' ? 'selected' : '' ?>>New Zealand</option>
            <option value="India" <?= $filters['country'] === 'India' ? 'selected' : '' ?>>India</option>
            <option value="Philippines" <?= $filters['country'] === 'Philippines' ? 'selected' : '' ?>>Philippines</option>
            <option value="Malaysia" <?= $filters['country'] === 'Malaysia' ? 'selected' : '' ?>>Malaysia</option>
            <option value="United Arab Emirates" <?= $filters['country'] === 'United Arab Emirates' ? 'selected' : '' ?>>United Arab Emirates</option>
            <option value="Bangladesh" <?= $filters['country'] === 'Bangladesh' ? 'selected' : '' ?>>Bangladesh</option>
            <option value="Qatar" <?= $filters['country'] === 'Qatar' ? 'selected' : '' ?>>Qatar</option>
            <option value="Nepal" <?= $filters['country'] === 'Nepal' ? 'selected' : '' ?>>Nepal</option>
        </select>
    </div>
    <button type="submit">Filter</button>
</form>

<table id="salesReportTable" class="display" style="width:100%">
    <thead><tr><th></th><th>Organization</th><th>Amount (from QBO)</th><th>Region</th><th>Quote Count</th><th>Pax</th><th>Country</th></tr></thead>
    <tfoot><tr><th></th><th>Page Total:</th><th></th><th></th><th></th><th></th><th></th></tr></tfoot>
</table>

<script>
    const reportData = <?= json_encode($final_report_data); ?>;
    function toggleCheckboxes(id) { const checkboxes = document.getElementById(id); if (checkboxes) { checkboxes.style.display = checkboxes.style.display === 'block' ? 'none' : 'block'; } }
    function updateSelectedOptions(prefix) { const checkboxContainer = document.getElementById(prefix + '-checkboxes'); if (!checkboxContainer) return; const selectedOptionsSpan = document.getElementById(prefix + '-selected-options'); const checkedCheckboxes = checkboxContainer.querySelectorAll('input[type="checkbox"]:checked'); if (checkedCheckboxes.length > 0) { let selectedTexts = []; checkedCheckboxes.forEach(cb => { selectedTexts.push(cb.parentElement.textContent.trim()); }); selectedOptionsSpan.textContent = selectedTexts.join(', '); } else { selectedOptionsSpan.textContent = 'Select ' + prefix.charAt(0).toUpperCase() + prefix.slice(1) + '(s)'; } }
    window.addEventListener('click', function(e) { document.querySelectorAll('.multiselect-container').forEach(container => { if (!container.contains(e.target)) { container.querySelector('.checkboxes').style.display = 'none'; } }); });
    
    $(document).ready(function() {
        const table = $('#salesReportTable').DataTable({
            data: reportData,
            columns: [ { className: 'dt-control', orderable: false, data: null, defaultContent: '' }, { data: 'organization_name' }, { data: 'qbo_total_amount', render: function(data) { return '$' + parseFloat(data).toFixed(2).replace(/\d(?=(\d{3})+\.)/g, '$&,'); }, className: 'currency' }, { data: 'region' }, { data: 'quote_count' }, { data: 'total_pax' }, { data: 'country' } ],
            order: [[1, 'asc']], paging: true, pageLength: 50, searching: true, info: true,
            footerCallback: function (row, data, start, end, display) { var api = this.api(); const total = (colIndex) => api.column(colIndex, { page: 'current' }).data().reduce((a, b) => a + parseFloat(b || 0), 0); $(api.column(1).footer()).html('Page Total:'); $(api.column(2).footer()).html('$' + total(2).toFixed(2).replace(/\d(?=(\d{3})+\.)/g, '$&,')); $(api.column(4).footer()).html(total(4)); $(api.column(5).footer()).html(total(5)); }
        });

        $('#salesReportTable tbody').on('click', 'td.dt-control', async function () {
            var tr = $(this).closest('tr'); var row = table.row(tr);
            if (row.child.isShown()) { row.child.hide(); tr.removeClass('dt-hasChild'); }
            else {
                tr.addClass('dt-hasChild'); row.child('<div>Loading details...</div>').show();
                const rowData = row.data(); const orgId = rowData.organizationid; const orgName = rowData.organization_name;
                const dateFrom = $('#date_from').val(); const dateTo = $('#date_to').val();
                try {
                    const apiUrl = `api_get_reconciliation_details.php?id=${orgId}&name=${encodeURIComponent(orgName)}&from=${dateFrom}&to=${dateTo}`;
                    const response = await fetch(apiUrl); const data = await response.json();
                    if (!response.ok || data.error) throw new Error(data.error || `API returned status: ${response.status}`);
                    const childHtml = formatChildRow(data); row.child(childHtml).show();
                } catch (error) { console.error('Failed to fetch details:', error); row.child(`<div style="color:red;">Failed to load details: ${error.message}</div>`).show(); }
            }
        });
        updateSelectedOptions('region');
    });

    function formatChildRow(data) {
        let invoiceHtml = '<h4>Invoice Breakdown (QBO)</h4>';
        if (data.invoice_details && Object.keys(data.invoice_details).length > 0) {
            invoiceHtml += '<table class="detail-table"><thead><tr><th>Invoice #</th><th>Quote #</th><th>Due Date</th><th>Open Balance</th><th>Status</th></tr></thead><tbody>';
            for (const invNum in data.invoice_details) {
                const inv = data.invoice_details[invNum];
                let statusClass = 'status-open'; let statusText = 'Open';
                if (inv.open_balance <= 0.01) { statusClass = 'status-paid'; statusText = 'Paid'; }
                else if (new Date(inv.due_date) < new Date()) { statusClass = 'status-overdue'; statusText = 'Overdue'; }
                invoiceHtml += `<tr><td>${invNum}</td><td>${inv.quote_no || 'N/A'}</td><td>${inv.due_date || 'N/A'}</td><td>$${parseFloat(inv.open_balance).toFixed(2)}</td><td class="${statusClass}">${statusText}</td></tr>`;
            }
            invoiceHtml += '</tbody></table>';
        } else { invoiceHtml += '<p>No open invoices found in QBO for this organization.</p>'; }
        let quoteHtml = '<h4>Quote Breakdown</h4>';
        if (data.quote_details && data.quote_details.length > 0) {
            quoteHtml += '<table class="detail-table"><thead><tr><th>Quote #</th><th>Travel Date</th><th>Status</th><th>Pax</th></tr></thead><tbody>';
            data.quote_details.forEach(item => { quoteHtml += `<tr><td><a href="quote.php?opt=summary&quoteNumber=${encodeURIComponent(item.quote_no)}" target="_blank">${item.quote_no}</a></td><td>${item.travel_date || 'N/A'}</td><td>${item.quotestage || 'N/A'}</td><td>${item.total_pax || '0'}</td></tr>`; });
            quoteHtml += '</tbody></table>';
        } else { quoteHtml += '<p>No quotes found for this organization in the selected date range.</p>'; }
        return `<div class="detail-container"><div class="detail-section">${invoiceHtml}</div><div class="detail-section">${quoteHtml}</div></div>`;
    }
</script>