<?php
// qbo_supplier_graph_report.php
// Provides a tabbed graphical interface for analyzing supplier expenses.

use QuickBooksOnline\API\DataService\DataService;

require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/qbo_functions.php';
require_once __DIR__ . '/config/qbo_config.php';
global $qboBaseConfig;

// --- QBO Connection & Initial Data Fetch ---
$dataService = null;
$allSuppliers = [];
$qbo_is_connected = false;

$tokenStorageFile = __DIR__ . '/tokens/qbo_token.json';
if (file_exists($tokenStorageFile)) {
    $currentTokens = json_decode(file_get_contents($tokenStorageFile), true);
    if ($currentTokens && !empty($currentTokens['access_token'])) {
        $qboBaseConfig['accessTokenKey'] = $currentTokens['access_token'];
        try {
            $dataService = DataService::Configure($qboBaseConfig);
            $qbo_is_connected = true;

            // Fetch all active vendors once for the dropdown filters
            $allSuppliersData = $dataService->Query("SELECT Id, DisplayName FROM Vendor WHERE Active = true ORDERBY DisplayName ASC MAXRESULTS 1000");
            if ($allSuppliersData) {
                foreach ($allSuppliersData as $supplier) {
                    $allSuppliers[] = $supplier->DisplayName;
                }
            }
        } catch (Exception $e) {
            $qbo_is_connected = false;
            echo "<!-- QBO Connection Error: " . htmlspecialchars($e->getMessage()) . " -->";
        }
    }
}

?>

<style>
    /* Using the same CSS as pnl report for a consistent look and feel */
    .tab-container { border-bottom: 2px solid #dee2e6; display: flex; flex-wrap: wrap; margin-bottom: 20px; }
    .tab-btn { background-color: transparent; border: none; padding: 12px 18px; font-size: 1.1em; cursor: pointer; position: relative; color: #6c757d; }
    .tab-btn.active { font-weight: bold; color: #005ea2; }
    .tab-btn.active::after { content: ''; position: absolute; bottom: -2px; left: 0; right: 0; height: 2px; background-color: #005ea2; }
    .tab-content { display: none; }
    .tab-content.active { display: block; }
    .comparison-container { display: grid; grid-template-columns: repeat(auto-fit, minmax(500px, 1fr)); gap: 20px; }
    .graph-instance { background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); margin-top: 20px; position: relative; }
    .graph-instance h3 { margin-top: 0; color: #333; padding-right: 30px; }
    .filter-form { margin-bottom: 15px; display: flex; gap: 15px; align-items: center; }
    .add-comparison-btn { background-color: #007bff; color: white; border: none; padding: 8px 15px; border-radius: 4px; cursor: pointer; font-size: 0.9em; margin-top: 20px; }
    .remove-comparison-btn { background-color: #dc3545; color: white; border: none; font-size: 1em; line-height: 1.2; cursor: pointer; padding: 2px 8px; border-radius: 50%; position: absolute; top: 15px; right: 15px; }
    .graph-canvas-wrapper { position: relative; height: 400px; }

    .loading-overlay { position: absolute; top: 0; left: 0; right: 0; bottom: 0; background: rgba(255, 255, 255, 0.85); display: flex; align-items: center; justify-content: center; z-index: 10; border-radius: 5px; transition: opacity 0.3s, visibility 0.3s; opacity: 0; visibility: hidden; }
    .loading-overlay.visible { opacity: 1; visibility: visible; }
    .spinner { width: 50px; height: 50px; border: 5px solid #f3f3f3; border-top: 5px solid #005ea2; border-radius: 50%; animation: spin 1s linear infinite; }
    @keyframes spin { 0% { transform: rotate(0deg); } 100% { transform: rotate(360deg); } }
</style>

<div class="report-header">
    <h2>Supplier Expense Analysis</h2>
    <p>Visual representation of spend and supplier performance.</p>
</div>

<div class="tab-container">
    <button class="tab-btn active" data-tab="monthly">Monthly Total</button>
    <button class="tab-btn" data-tab="yearly">Yearly Total</button>
    <button class="tab-btn" data-tab="top_10">Top 10 Suppliers</button>
    <button class="tab-btn" data-tab="performance">Supplier Performance</button>
</div>

<div id="monthly" class="tab-content active"><div id="monthly-graphs" class="comparison-container"></div><button class="add-comparison-btn" data-view="monthly">+ Add Comparison</button></div>
<div id="yearly" class="tab-content"><div id="yearly-graphs" class="comparison-container"></div></div>
<div id="top_10" class="tab-content"><div id="top_10-graphs" class="comparison-container"></div></div>
<div id="performance" class="tab-content"><div id="performance-graphs" class="comparison-container"></div><button class="add-comparison-btn" data-view="performance">+ Add Comparison</button></div>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const qboIsConnected = <?php echo json_encode($qbo_is_connected); ?>;
    const allSuppliers = <?php echo json_encode($allSuppliers); ?>;
    const chartInstances = {};

    if (!qboIsConnected) {
        document.body.innerHTML = '<h1>QuickBooks Online connection not found. Please connect to QBO first.</h1>';
        return;
    }

    const generateYearSelector = (graphId) => {
        const now = new Date();
        let optionsHtml = '';
        for (let i = 0; i <= 5; i++) {
            const year = now.getFullYear() - i;
            optionsHtml += `<option value="${year}" ${i === 0 ? 'selected' : ''}>${year}</option>`;
        }
        return `<label>For Year: <select class="filter-selector" data-filter-type="year" data-graph-id="${graphId}">${optionsHtml}</select></label>`;
    };

    const generateSupplierSelector = (graphId) => {
        let optionsHtml = allSuppliers.map(s => `<option value="${s}">${s}</option>`).join('');
        return `<label>Supplier: <select class="filter-selector" data-filter-type="supplier" data-graph-id="${graphId}">${optionsHtml}</select></label>`;
    };
    
    const generatePeriodSelector = (graphId) => {
        const options = ['Monthly', 'Quarterly', 'Yearly'];
        let optionsHtml = options.map(o => `<option value="${o}">${o}</option>`).join('');
        return `<label>Period: <select class="filter-selector" data-filter-type="period" data-graph-id="${graphId}">${optionsHtml}</select></label>`;
    }

    async function fetchDataForGraph(reportType, year = null, supplierName = null, period = 'Monthly') {
        let url = `qbo_ajax_supplier_graph_data.php?report_type=${reportType}`;
        if (year) url += `&year=${year}`;
        if (supplierName) url += `&supplier_name=${encodeURIComponent(supplierName)}`;
        if (period) url += `&period=${period}`;
        
        const response = await fetch(url);
        const result = await response.json();
        return result.success ? result.data : null;
    }

    async function updateGraph(graphId, view) {
        const canvas = document.getElementById(graphId);
        if (!canvas) return;
        const wrapper = canvas.closest('.graph-instance');
        const loadingOverlay = wrapper.querySelector('.loading-overlay');
        loadingOverlay.classList.add('visible');

        const yearSelector = wrapper.querySelector('.filter-selector[data-filter-type="year"]');
        const supplierSelector = wrapper.querySelector('.filter-selector[data-filter-type="supplier"]');
        const periodSelector = wrapper.querySelector('.filter-selector[data-filter-type="period"]');

        const year = yearSelector ? yearSelector.value : new Date().getFullYear();
        const supplierName = supplierSelector ? supplierSelector.value : null;
        const period = periodSelector ? periodSelector.value : 'Monthly';

        const data = await fetchDataForGraph(view, year, supplierName, period);
        loadingOverlay.classList.remove('visible');

        const ctx = canvas.getContext('2d');
        if (!data || !data.labels || data.labels.length === 0) {
            if (chartInstances[graphId]) { chartInstances[graphId].destroy(); delete chartInstances[graphId]; }
            ctx.clearRect(0, 0, canvas.width, canvas.height);
            ctx.font = "16px Arial"; ctx.textAlign = "center";
            ctx.fillText("No data found for this period.", canvas.width / 2, 50);
            return;
        }
        
        const chartType = view === 'top_10' ? 'bar' : 'bar'; // Use bar chart for most, could be 'line' for trends
        const indexAxis = view === 'top_10' ? 'y' : 'x'; // Horizontal for Top 10

        const chartConfig = {
            type: chartType,
            data: {
                labels: data.labels,
                datasets: [{
                    label: 'Total Expenses',
                    data: data.values,
                    backgroundColor: 'rgba(211, 84, 0, 0.7)',
                    borderColor: 'rgba(211, 84, 0, 1)',
                    borderWidth: 1
                }]
            },
            options: {
                indexAxis: indexAxis,
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { display: view !== 'top_10' },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                let val = indexAxis === 'y' ? context.parsed.x : context.parsed.y;
                                return ' ' + new Intl.NumberFormat('en-US', { style: 'currency', currency: 'USD' }).format(val);
                            }
                        }
                    }
                },
                scales: {
                    x: { ticks: { callback: value => '$' + value.toLocaleString() } },
                    y: { ticks: { callback: function(value, index) { 
                               return indexAxis === 'y' ? this.getLabelForValue(value) : '$' + value.toLocaleString();
                           }}}
                }
            }
        };
        // Adjust scales for horizontal vs vertical charts
        if(indexAxis === 'x') {
            chartConfig.options.scales.x.ticks.callback = val => val; // Keep label as is
            chartConfig.options.scales.y.ticks.callback = val => '$' + val.toLocaleString();
        } else {
            chartConfig.options.scales.x.ticks.callback = val => '$' + val.toLocaleString();
            chartConfig.options.scales.y.ticks.callback = function(val) { return this.getLabelForValue(val); };
        }


        if (chartInstances[graphId]) {
            chartInstances[graphId].destroy();
        }
        chartInstances[graphId] = new Chart(ctx, chartConfig);
    }

    function createGraphInstance(view, isFirst = false) {
        const graphContainer = document.getElementById(`${view}-graphs`);
        const instanceId = `${view}-${Date.now()}`;
        const canvasId = `chart-${instanceId}`;
        const graphDiv = document.createElement('div');
        graphDiv.className = 'graph-instance';

        let titleText = '';
        let filterHtml = '<div class="filter-form">';

        switch(view) {
            case 'monthly':
                titleText = isFirst ? 'Monthly Total Expenses' : 'Comparison Year';
                filterHtml += generateYearSelector(canvasId);
                break;
            case 'yearly':
                titleText = 'Yearly Total Expenses (Last 7 Years)';
                break;
            case 'top_10':
                titleText = 'Top 10 Suppliers (Current Year)';
                filterHtml += generateYearSelector(canvasId);
                break;
            case 'performance':
                titleText = isFirst ? 'Supplier Performance' : 'Compare Supplier';
                filterHtml += generateSupplierSelector(canvasId);
                filterHtml += generateYearSelector(canvasId);
                filterHtml += generatePeriodSelector(canvasId);
                break;
        }
        filterHtml += '</div>';

        graphDiv.innerHTML = `
            <h3>${titleText} <button class="remove-comparison-btn" style="display: ${isFirst ? 'none' : 'inline-block'};">×</button></h3>
            ${filterHtml}
            <div class="graph-canvas-wrapper">
                <div class="loading-overlay"><div class="spinner"></div></div>
                <canvas id="${canvasId}"></canvas>
            </div>`;
        
        graphContainer.appendChild(graphDiv);

        graphDiv.querySelectorAll('.filter-selector').forEach(sel => {
            sel.addEventListener('change', () => updateGraph(canvasId, view));
        });

        updateGraph(canvasId, view);
    }

    document.querySelectorAll('.comparison-container').forEach(container => {
        container.addEventListener('click', function(e) {
            if (e.target.classList.contains('remove-comparison-btn')) {
                const instanceDiv = e.target.closest('.graph-instance');
                const canvasId = instanceDiv.querySelector('canvas').id;
                if (chartInstances[canvasId]) {
                    chartInstances[canvasId].destroy();
                    delete chartInstances[canvasId];
                }
                instanceDiv.remove();
            }
        });
    });

    const tabs = document.querySelectorAll('.tab-btn');
    tabs.forEach(tab => {
        tab.addEventListener('click', () => {
            tabs.forEach(t => t.classList.remove('active'));
            document.querySelectorAll('.tab-content').forEach(c => c.classList.remove('active'));

            tab.classList.add('active');
documents.getElementById(tab.dataset.tab).classList.add('active');
        });
    });

    document.querySelectorAll('.add-comparison-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            createGraphInstance(this.dataset.view, false);
        });
    });

    // Initial Load
    createGraphInstance('monthly', true);
    createGraphInstance('yearly', true);
    createGraphInstance('top_10', true);
    createGraphInstance('performance', true);
});
</script>