<?php
// qbo_supplier_expense_report.php
// FINAL, CORRECTED VERSION: Adds the missing report period filter.

use QuickBooksOnline\API\DataService\DataService;
use QuickBooksOnline\API\ReportService\ReportService;

require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/qbo_functions.php';
require_once __DIR__ . '/config/qbo_config.php';
global $qboBaseConfig;

// --- Report Filters & Variables ---
$finYearStart = (date('m') < 7) ? date('Y-07-01', strtotime('-1 year')) : date('Y-07-01');
$startDate = isset($_GET['start_date']) && !empty($_GET['start_date']) ? $_GET['start_date'] : $finYearStart;
$endDate = isset($_GET['end_date']) && !empty($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-d');
$accountingMethod = isset($_GET['accounting_method']) ? $_GET['accounting_method'] : 'Accrual';

$dataService = null; $report = null;

// --- QBO Connection & Token Refresh ---
$tokenStorageFile = __DIR__ . '/tokens/qbo_token.json';
$currentTokens = file_exists($tokenStorageFile) ? json_decode(file_get_contents($tokenStorageFile), true) : null;
if ($currentTokens && !empty($currentTokens['access_token'])) {
    $qboBaseConfig['accessTokenKey'] = $currentTokens['access_token'];
    try { $dataService = DataService::Configure($qboBaseConfig); } catch (Exception $e) { $dataService = null; }
} else { exit("QBO Authentication failed."); }

$reportFetchSuccess = false;
if ($dataService) {
    try {
        $reportService = new ReportService($dataService->getServiceContext());
        $reportService->setStartDate($startDate);
        $reportService->setEndDate($endDate);
        $reportService->setAccountingMethod($accountingMethod);
        $report = $reportService->executeReport("VendorExpenses");
        if ($report) { $reportFetchSuccess = true; }
    } catch (Exception $e) { /* Suppress error */ }
}
?>
<div class="report-header">
    <h2>Expenses by Supplier Summary</h2>
    <p>For period: <?php echo htmlspecialchars(date("F j, Y", strtotime($startDate))) . " - " . htmlspecialchars(date("F j, Y", strtotime($endDate))); ?> </p>
</div>
<form method="get" class="filter-form">
    <input type="hidden" name="report" value="supplier_expense">
    <div>
        <label>Report period:</label>
        <select id="report_period_selector">
            <option value="This_Fiscal_Year" selected>This Financial Year</option><option value="Last_Fiscal_Year">Last Financial Year</option>
            <option value="This_Month">This Month</option><option value="Last_Month">Last Month</option>
            <option value="custom">Custom</option>
        </select>
    </div>
    <div><label>From:</label><input type="date" name="start_date" id="start_date" value="<?php echo htmlspecialchars($startDate); ?>"></div>
    <div><label>To:</label><input type="date" name="end_date" id="end_date" value="<?php echo htmlspecialchars($endDate); ?>"></div>
    <div><label>Method:</label><select name="accounting_method" onchange="this.form.submit()"><option value="Accrual" <?php if($accountingMethod == 'Accrual') echo 'selected';?>>Accrual</option><option value="Cash" <?php if($accountingMethod == 'Cash') echo 'selected';?>>Cash</option></select></div>
    <button type="submit">Run Report</button>
</form>
<table>
    <thead><tr><th>Supplier</th><th class="currency">Amount</th></tr></thead>
    <tbody>
        <?php
        if ($reportFetchSuccess && isset($report->Rows->Row)) {
            $rows = is_array($report->Rows->Row) ? $report->Rows->Row : [$report->Rows->Row];
            foreach ($rows as $row) {
                if (isset($row->ColData)) {
                    $vendorName = htmlspecialchars($row->ColData[0]->value ?? 'N/A');
                    $vendorId = $row->ColData[0]->id ?? null;
                    $amount = (float)($row->ColData[1]->value ?? 0);
                    if ($amount == 0 || !$vendorId) continue;
                    $detailUrl = "?report=supplier_expense_detail&vendor_id={$vendorId}&start_date={$startDate}&end_date={$endDate}&accounting_method={$accountingMethod}&vendor_name=" . urlencode($vendorName);
                    echo "<tr><td><strong><a href='{$detailUrl}'>{$vendorName}</a></strong></td><td class='currency'>" . number_format($amount, 2) . "</td></tr>";
                }
            }
        } else { echo '<tr><td colspan="2" class="no-data-message">No expense data found for the selected period.</td></tr>'; }
        ?>
    </tbody>
    <tfoot>
        <?php
        if ($reportFetchSuccess && isset($report->Summary->ColData)) {
            $totalAmount = (float)($report->Summary->ColData[1]->value ?? 0);
            echo '<tr class="total-row"><td><strong>TOTAL</strong></td><td class="currency"><strong>' . number_format($totalAmount, 2) . '</strong></td></tr>';
        }
        ?>
    </tfoot>
</table>
<script>
    // YOUR CORRECTED DATE LOGIC
    document.getElementById('report_period_selector')?.addEventListener('change', function() {
        const selectedPeriod = this.value; const today = new Date();
        let startDate, endDate;
        const formatDate = (d) => { const pad = (num) => num.toString().padStart(2, '0'); return `${d.getFullYear()}-${pad(d.getMonth() + 1)}-${pad(d.getDate())}`; }
        const finYearStartMonth = 6;
        switch (selectedPeriod) {
            case 'This_Month': startDate = new Date(today.getFullYear(), today.getMonth(), 1); endDate = new Date(today.getFullYear(), today.getMonth() + 1, 0); break;
            case 'This_Fiscal_Year':
                let finYear = today.getFullYear(); if (today.getMonth() < finYearStartMonth) finYear--;
                startDate = new Date(finYear, finYearStartMonth, 1); endDate = new Date(finYear + 1, finYearStartMonth, 0); break;
            case 'Last_Month': startDate = new Date(today.getFullYear(), today.getMonth() - 1, 1); endDate = new Date(today.getFullYear(), today.getMonth(), 0); break;
            case 'Last_Fiscal_Year':
                let lastFinYear = today.getFullYear(); if (today.getMonth() < finYearStartMonth) lastFinYear--;
                startDate = new Date(lastFinYear - 1, finYearStartMonth, 1); endDate = new Date(lastFinYear, finYearStartMonth, 0); break;
            case 'custom': default: return;
        }
        if(startDate && endDate){
            document.getElementById('start_date').value = formatDate(startDate);
            document.getElementById('end_date').value = formatDate(endDate);
        }
    });
</script>