<?php
// qbo_test_query.php
// A simple page to test arbitrary QBO API queries.

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// --- Essential Includes ---
// Ensure qbo_functions.php does NOT include vendor/autoload.php again if this script does.
require_once __DIR__ . '/../vendor/autoload.php'; // QBO SDK Autoloader - Adjust path if this is not project root
require_once __DIR__ . '/qbo_functions.php';      // Your QBOUtilityLibrary

use QuickBooksOnline\API\Core\OAuth\OAuth2\OAuth2LoginHelper;
use QuickBooksOnline\API\Exception\SdkException;
use QuickBooksOnline\API\Exception\ServiceException;
use QuickBooksOnline\API\Exception\IdsException;


// --- Configuration Paths (ADJUST THESE PATHS TO MATCH YOUR SETUP) ---
$qboBaseConfigFile = __DIR__ . '/config/qbo_config.php';   // Path to QBO base config (ClientID, Secret, RealmID, BaseURL)
$qboTokenStorageFile = __DIR__ . '/tokens/qbo_token.json'; // Path to store/read QBO OAuth2 tokens

// --- Global Variables & Initialization ---
$qboUtil = null;
$qboErrorMessage = '';
$qboBaseConfig = null;
$query_results = null;
$query_error = null;
$submitted_query = '';

// --- QBO Token Management Functions (Copied from testing_ground.php for standalone use) ---
if (file_exists($qboBaseConfigFile)) {
    require_once $qboBaseConfigFile; // Defines $qboBaseConfig
} else {
    $qboErrorMessage = "CRITICAL: QBO base configuration file not found: " . htmlspecialchars($qboBaseConfigFile);
    error_log("qbo_test_query.php: " . $qboErrorMessage);
}

function loadQBOTokensTestQuery(): ?array { // Renamed to avoid conflict if included elsewhere
    global $qboTokenStorageFile;
    if (file_exists($qboTokenStorageFile)) {
        $json = file_get_contents($qboTokenStorageFile);
        if ($json === false) { error_log("qbo_test_query.php: Failed to read token file: " . $qboTokenStorageFile); return null; }
        $tokens = json_decode($json, true);
        if (json_last_error() !== JSON_ERROR_NONE) { error_log("qbo_test_query.php: Failed to decode JSON from token file: " . json_last_error_msg()); return null; }
        return $tokens;
    }
    error_log("qbo_test_query.php: Token storage file not found: " . $qboTokenStorageFile);
    return null;
}

function saveQBOTokensTestQuery(string $accessToken, string $refreshToken): bool { // Renamed
    global $qboTokenStorageFile;
    $tokens = ['access_token' => $accessToken, 'refresh_token' => $refreshToken, 'last_updated' => date('Y-m-d H:i:s')];
    $tokenDir = dirname($qboTokenStorageFile);
    if (!is_dir($tokenDir)) {
        if (!mkdir($tokenDir, 0755, true) && !is_dir($tokenDir)) { error_log("qbo_test_query.php: Could not create token directory: " . $tokenDir); return false; }
    }
    if (file_put_contents($qboTokenStorageFile, json_encode($tokens, JSON_PRETTY_PRINT))) {
        error_log("qbo_test_query.php: Tokens saved to: " . $qboTokenStorageFile);
        return true;
    }
    error_log("qbo_test_query.php: Could not write tokens to file: " . $qboTokenStorageFile);
    return false;
}

// --- QBO Initialization ---
if ($qboBaseConfig) {
    $currentTokens = loadQBOTokensTestQuery();
    $qboConfigForLibrary = $qboBaseConfig;

    if ($currentTokens && !empty($currentTokens['refresh_token'])) {
        $qboConfigForLibrary['refreshTokenKey'] = $currentTokens['refresh_token'];
        try {
            $oauth2LoginHelper = new OAuth2LoginHelper($qboBaseConfig['ClientID'], $qboBaseConfig['ClientSecret']);
            error_log("qbo_test_query.php: Attempting QBO token refresh...");
            $refreshedAccessTokenObj = $oauth2LoginHelper->refreshAccessTokenWithRefreshToken($qboConfigForLibrary['refreshTokenKey']);
            $newAccessToken = $refreshedAccessTokenObj->getAccessToken();
            $newRefreshToken = $refreshedAccessTokenObj->getRefreshToken();
            if (saveQBOTokensTestQuery($newAccessToken, $newRefreshToken)) {
                error_log("qbo_test_query.php: SUCCESS: QBO Tokens refreshed and saved.");
                $currentTokens['access_token'] = $newAccessToken;
                $currentTokens['refresh_token'] = $newRefreshToken;
            } else {
                $qboErrorMessage = "ERROR: QBO Tokens refreshed but failed to save."; error_log("qbo_test_query.php: " . $qboErrorMessage);
            }
        } catch (SdkException $e) {
            $qboErrorMessage = "ERROR refreshing QBO tokens (SDK): " . $e->getMessage(); error_log("qbo_test_query.php: " . $qboErrorMessage);
        } catch (ServiceException $e) {
            $qboErrorMessage = "ERROR refreshing QBO tokens (Service): " . $e->getMessage() . " OAuth Helper: " . $e->getOAuthHelperError(); error_log("qbo_test_query.php: " . $qboErrorMessage . " Details: " . $e->getResponseBody());
        } catch (Exception $e) {
            $qboErrorMessage = "Unexpected error during QBO token refresh: " . $e->getMessage(); error_log("qbo_test_query.php: " . $qboErrorMessage);
        }
    } elseif (!$currentTokens) {
        $qboErrorMessage = "QBO tokens not found. Refresh cannot be attempted. Initial OAuth2 flow may be needed."; error_log("qbo_test_query.php: " . $qboErrorMessage);
    } else {
         $qboErrorMessage = "Warning: QBO Refresh token not found. Cannot refresh. Using existing access token if available."; error_log("qbo_test_query.php: " . $qboErrorMessage);
    }

    if ($currentTokens && !empty($currentTokens['access_token'])) {
        $qboConfigForLibrary['accessTokenKey'] = $currentTokens['access_token'];
        $qboConfigForLibrary['refreshTokenKey'] = $currentTokens['refresh_token'] ?? $qboConfigForLibrary['refreshTokenKey'] ?? null;
    } else {
        if (empty($qboErrorMessage)) $qboErrorMessage = "CRITICAL: No valid QBO access token. QBO operations disabled.";
        error_log("qbo_test_query.php: " . $qboErrorMessage);
    }

    if (empty($qboErrorMessage) && !empty($qboConfigForLibrary['accessTokenKey'])) {
        try {
            // SDK Log Location (optional)
            if (isset($qboConfigForLibrary['logLocation'])) {
                $sdkLogDir = $qboConfigForLibrary['logLocation'];
                 if (!is_dir($sdkLogDir)) { if (!mkdir($sdkLogDir, 0755, true) && !is_dir($sdkLogDir)) { unset($qboConfigForLibrary['logLocation']); }}
                 elseif (!is_writable($sdkLogDir)) { unset($qboConfigForLibrary['logLocation']);}
            }
            $qboLibrary = new QBOUtilityLibrary($qboConfigForLibrary); // Instantiate your library
            $dataService = $qboLibrary->getDataService(); // Get the configured DataService instance
            error_log("qbo_test_query.php: QBOUtilityLibrary and DataService instantiated successfully.");
        } catch (InvalidArgumentException $e) {
            $qboErrorMessage = "QBOUtilityLibrary Config Error: " . $e->getMessage(); error_log("qbo_test_query.php: " . $qboErrorMessage);
        } catch (Exception $e) {
            $qboErrorMessage = "Error instantiating QBOUtilityLibrary/DataService: " . $e->getMessage(); error_log("qbo_test_query.php: " . $qboErrorMessage);
        }
    }
} elseif (empty($qboErrorMessage)) {
    $qboErrorMessage = "QBO base configuration not loaded. QBO features disabled.";
    error_log("qbo_test_query.php: " . $qboErrorMessage);
}

// --- Handle Query Submission ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['qbo_query'])) {
    $submitted_query = trim($_POST['qbo_query']);
    if (empty($submitted_query)) {
        $query_error = "Query cannot be empty.";
    } elseif (!$dataService) {
        $query_error = "QBO DataService is not available. Check configuration and token errors above. " . htmlspecialchars($qboErrorMessage);
    } else {
        try {
            error_log("qbo_test_query.php: Executing QBO Query: " . $submitted_query);
            $query_results = $dataService->Query($submitted_query);
            if ($query_results === null || (is_array($query_results) && empty($query_results))) {
                 error_log("qbo_test_query.php: Query executed, but returned no results or null.");
            } else {
                 error_log("qbo_test_query.php: Query executed successfully, " . (is_array($query_results) ? count($query_results) : '1') . " result(s) returned.");
            }
        } catch (IdsException $e) {
            $query_error = "QBO API IdsException: " . $e->getMessage();
            error_log("qbo_test_query.php: IdsException during query: " . $e->getMessage());
        } catch (Exception $e) {
            $query_error = "General Exception during query: " . $e->getMessage();
            error_log("qbo_test_query.php: General Exception during query: " . $e->getMessage());
        }
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>QBO API Query Tester</title>
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; margin: 20px; background-color: #f4f4f4; color: #333; }
        .container { max-width: 900px; margin: auto; background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 0 10px rgba(0,0,0,0.1); }
        h1 { color: #005ea2; text-align: center; }
        textarea { width: 98%; min-height: 100px; padding: 10px; margin-bottom: 15px; border: 1px solid #ccc; border-radius: 4px; font-family: monospace; font-size: 14px; }
        input[type="submit"] { background-color: #005ea2; color: white; padding: 10px 20px; border: none; border-radius: 4px; cursor: pointer; font-size: 16px; }
        input[type="submit"]:hover { background-color: #004a80; }
        .results, .error { margin-top: 20px; padding: 15px; border-radius: 4px; }
        .results pre { background-color: #e9ecef; padding: 15px; border-radius: 4px; white-space: pre-wrap; word-wrap: break-word; max-height: 600px; overflow-y: auto; border: 1px solid #ced4da; }
        .error { background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        .status-message { background-color: #fff3cd; color: #856404; border: 1px solid #ffeeba; padding: 10px; margin-bottom:15px; border-radius: 4px; }
        label { display: block; margin-bottom: 5px; font-weight: bold; }
    </style>
</head>
<body>
    <div class="container">
        <h1>QBO API Query Tester</h1>

        <?php if (!empty($qboErrorMessage)): ?>
            <div class="status-message error">
                <strong>QBO Initialization Status:</strong> <?php echo htmlspecialchars($qboErrorMessage); ?>
                 <?php if (strpos(strtolower($qboErrorMessage), 'token') !== false || strpos(strtolower($qboErrorMessage), 'oauth') !== false) : ?>
                    <br><em>Ensure QBO tokens are correctly configured (<code><?php echo htmlspecialchars($qboTokenStorageFile); ?></code>) and base config (<code><?php echo htmlspecialchars($qboBaseConfigFile); ?></code>) is valid. An initial OAuth2 authorization might be required if tokens are missing or invalid.</em>
                <?php endif; ?>
            </div>
        <?php elseif (isset($dataService)): ?>
            <div class="status-message" style="background-color: #d4edda; color: #155724; border-color: #c3e6cb;">QBO DataService Initialized Successfully. Ready to query.</div>
        <?php else: ?>
             <div class="status-message error">QBO DataService not initialized. Check logs and configuration.</div>
        <?php endif; ?>

        <form action="qbo_test_query.php" method="post">
            <div>
                <label for="qbo_query">Enter QBO Query:</label>
                <textarea name="qbo_query" id="qbo_query" placeholder="Example: SELECT * FROM Customer MAXRESULTS 5
SELECT * FROM Bill WHERE VendorRef = 'VendorID' AND ClassRef = 'ClassID' MAXRESULTS 10 (Note: ClassRef on Bill directly is not supported, this is just an example of syntax)"><?php echo htmlspecialchars($submitted_query); ?></textarea>
            </div>
            <div>
                <input type="submit" value="Execute Query">
            </div>
        </form>

        <?php if ($query_error): ?>
            <div class="error">
                <h2>Query Error:</h2>
                <p><?php echo $query_error; // Already HTML formatted if IdsException ?></p>
            </div>
        <?php endif; ?>

        <?php if ($query_results !== null): ?>
            <div class="results">
                <h2>Query Results:</h2>
                <?php if (empty($query_results) && is_array($query_results)): ?>
                    <p>Query executed successfully, but returned no results.</p>
                <?php else: ?>
                    <pre><?php echo htmlspecialchars(json_encode($query_results, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)); ?></pre>
                <?php endif; ?>
            </div>
        <?php elseif ($_SERVER['REQUEST_METHOD'] === 'POST' && empty($query_error) && $submitted_query): ?>
             <div class="results">
                <h2>Query Results:</h2>
                <p>Query may have executed, but no results were returned or the result was null. Check QBO logs if unexpected.</p>
            </div>
        <?php endif; ?>

    </div>
</body>
</html>