<?php
// qbo_transaction_explorer.php
// This script provides a filterable list of posted transactions, styled to look like the QBO banking feed.

// Use required QBO SDK classes
use QuickBooksOnline\API\Core\OAuth\OAuth2\OAuth2LoginHelper;
use QuickBooksOnline\API\DataService\DataService;

// --- Load required files ---
require_once __DIR__ . '/dbconn.php';
require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/qbo_functions.php';
require_once __DIR__ . '/config/qbo_config.php';

global $conn;
global $qboBaseConfig;

// --- Helper Functions ---
$tokenStorageFile = __DIR__ . '/tokens/qbo_token.json';
if (!function_exists('reportLog')) { function reportLog($message) { $timestamp = date('Y-M-d H:i:s'); echo "<!-- " . htmlspecialchars("[$timestamp] " . $message . "\n") . " -->\n"; } }
if (!function_exists('loadTokens')) { function loadTokens() { global $tokenStorageFile; if (file_exists($tokenStorageFile)) { $json = file_get_contents($tokenStorageFile); $tokens = json_decode($json, true); return json_last_error() === JSON_ERROR_NONE ? $tokens : null; } return null; } }
if (!function_exists('saveTokens')) { function saveTokens($accessToken, $refreshToken) { global $tokenStorageFile; $tokens = ['access_token' => $accessToken, 'refresh_token' => $refreshToken, 'last_updated' => date('Y-m-d H:i:s')]; $tokenDir = dirname($tokenStorageFile); if (!is_dir($tokenDir)) { mkdir($tokenDir, 0755, true); } return file_put_contents($tokenStorageFile, json_encode($tokens, JSON_PRETTY_PRINT)); } }
if (!function_exists('getQBOTransactionLink')) {
    function getQBOTransactionLink($txnId, $entityType, $baseUrl) {
        if (empty($txnId) || empty($entityType)) return '#';
        
        $entityType = strtolower($entityType);
        $linkPath = 'expense'; // Default path

        if ($entityType === 'invoice') $linkPath = 'invoice';
        elseif ($entityType === 'bill') $linkPath = 'bill';
        elseif ($entityType === 'payment') $linkPath = 'recvpayment';
        elseif ($entityType === 'salesreceipt') $linkPath = 'salesreceipt';
        elseif ($entityType === 'check') $linkPath = 'check';
        elseif ($entityType === 'deposit') $linkPath = 'deposit';
        elseif ($entityType === 'billpayment') $linkPath = 'billpayment';

        $isSandbox = (strcasecmp($baseUrl, 'Development') == 0);
        $linkBase = $isSandbox 
            ? "https://app.sandbox.qbo.intuit.com/app/{$linkPath}?txnId=" 
            : "https://app.qbo.intuit.com/app/{$linkPath}?txnId=";
            
        return $linkBase . htmlspecialchars($txnId);
    }
}

// --- Data Fetching Logic ---
$dataService = null;
$allTransactions = [];
$accounts = [];
$entityNameMap = []; // For Vendor and Customer names
$accountNameMap = []; // For Account/Category names
$error_message = null;

// --- Handle Filter Inputs ---
$defaultStartDate = date('Y-m-d', strtotime('-30 days'));
$defaultEndDate = date('Y-m-d');
$startDate = $_GET['start_date'] ?? $defaultStartDate;
$endDate = $_GET['end_date'] ?? $defaultEndDate;

// --- Initialize QBO DataService & Refresh Token ---
$currentTokens = loadTokens();
if ($currentTokens && !empty($currentTokens['refresh_token'])) {
    try {
        $oauth2LoginHelper = new OAuth2LoginHelper($qboBaseConfig['ClientID'], $qboBaseConfig['ClientSecret']);
        $refreshedAccessTokenObj = $oauth2LoginHelper->refreshAccessTokenWithRefreshToken($currentTokens['refresh_token']);
        $newAccessToken = $refreshedAccessTokenObj->getAccessToken();
        $newRefreshToken = $refreshedAccessTokenObj->getRefreshToken();
        if ($newAccessToken && $newRefreshToken && saveTokens($newAccessToken, $newRefreshToken)) {
            $currentTokens['access_token'] = $newAccessToken;
        }
    } catch (Exception $e) { $error_message = "Error refreshing tokens: " . $e->getMessage(); }
}

// --- Configure DataService with Access Token ---
if (!$error_message && !empty($currentTokens['access_token']) && !empty($qboBaseConfig['QBORealmID'])) {
    try {
        $qboBaseConfig['accessTokenKey'] = $currentTokens['access_token'];
        $dataService = DataService::Configure($qboBaseConfig);
        reportLog("DataService instantiated for Transaction Explorer.");
    } catch (Exception $e) { $error_message = "Error initializing QBO DataService: " . $e->getMessage(); }
} elseif(!$error_message) { $error_message = "QBO Authentication failed or realm ID is missing."; }

// --- Query for Data if DataService is ready ---
if ($dataService) {
    try {
        // Fetch Bank accounts for header cards
        $accounts = $dataService->Query("SELECT * FROM Account WHERE AccountType IN ('Bank', 'Credit Card') AND Active = true ORDERBY Name");

        // Pre-fetch look-up maps
        $vendors = $dataService->Query("SELECT Id, DisplayName FROM Vendor MAXRESULTS 1000");
        if ($vendors) { foreach ($vendors as $vendor) { $entityNameMap[$vendor->Id] = $vendor->DisplayName; } }
        
        $customers = $dataService->Query("SELECT Id, DisplayName FROM Customer MAXRESULTS 1000");
        if ($customers) { foreach ($customers as $customer) { $entityNameMap[$customer->Id] = $customer->DisplayName; } }

        $allAccounts = $dataService->Query("SELECT Id, Name FROM Account MAXRESULTS 1000");
        if ($allAccounts) { foreach ($allAccounts as $account) { $accountNameMap[$account->Id] = $account->Name; } }

        // Query all relevant transaction types
        $entityTypesToQuery = ['Purchase', 'Payment', 'SalesReceipt', 'BillPayment', 'Check', 'Deposit'];
        
        foreach ($entityTypesToQuery as $entity) {
            $query = "SELECT * FROM {$entity} WHERE TxnDate >= '{$startDate}' AND TxnDate <= '{$endDate}' MAXRESULTS 500";
            
            reportLog("Executing QBO Query: {$query}");
            $results = $dataService->Query($query);
            
            $queryError = $dataService->getLastError();
            if ($queryError) {
                reportLog("QBO Query Error for {$entity}: " . $queryError->getResponseBody());
            } elseif (!empty($results)) {
                $allTransactions = array_merge($allTransactions, $results);
            }
        }
        
        // Sort all transactions by date
        if (!empty($allTransactions)) {
            usort($allTransactions, function($a, $b) { return strtotime($b->TxnDate) - strtotime($a->TxnDate); });
        }

    } catch (Exception $e) { $error_message = "Exception during QBO Query: " . $e->getMessage(); }
}
?>

<!-- Main QBO-style Header -->
<div class="qbo-header-bar">
    <div class="account-title">
        <h2>Banking</h2>
    </div>
</div>

<!-- Account Cards Section -->
<section class="account-cards-container">
    <?php if (!empty($accounts)): ?>
        <?php foreach ($accounts as $index => $account): ?>
            <div class="account-card <?php echo ($index === 0) ? 'active' : ''; ?>" data-id="<?php echo htmlspecialchars($account->Id); ?>">
                <div class="card-title">
                    <span><?php echo htmlspecialchars($account->Name); ?></span>
                </div>
                <div class="card-body">
                    <p class="bank-balance">Bank: <?php echo htmlspecialchars(number_format($account->CurrentBalance, 2)); ?></p>
                    <p class="qbo-balance">In QuickBooks: <?php echo htmlspecialchars(number_format($account->CurrentBalanceWithSubAccounts, 2)); ?></p>
                    <p style="color: #777; font-size: 0.8em;">Standard Feed</p>
                </div>
            </div>
        <?php endforeach; ?>
    <?php else: ?>
        <p>No bank accounts found or unable to connect to QBO.</p>
    <?php endif; ?>
</section>

<!-- Transaction Tabs -->
<div class="transaction-tabs">
    <div class="tab" onclick="alert('Pending/Excluded views are not available via the API.');">Pending</div>
    <div class="tab active">Posted</div>
    <div class="tab" onclick="alert('Pending/Excluded views are not available via the API.');">Excluded</div>
</div>

<!-- Filter Form -->
<form action="" method="GET" class="filter-form">
    <input type="hidden" name="report" value="transactions">
    <!--
    <div class="search-box">
         <input type="search" name="search" placeholder="Search in memo..." value="" disabled>
    </div>
    -->
    <input type="date" id="start_date" name="start_date" value="<?php echo htmlspecialchars($startDate); ?>" class="date-filter" title="Start Date">
    <input type="date" id="end_date" name="end_date" value="<?php echo htmlspecialchars($endDate); ?>" class="date-filter" title="End Date">
    <button type="submit" title="Apply Filters">✔️</button>
    <button type="button" onclick="window.location.href='?report=transactions'" title="Reset Filters">🔄</button>
</form>

<!-- Transaction Results Table -->
<section id="transaction-list">
    <table>
        <thead>
            <tr>
                <th>Date</th>
                <th>Bank Description</th>
                <th class="currency">Spent</th>
                <th class="currency">Received</th>
                <th>From/To</th>
                <th>Match/Category</th>
                <th>Transaction Posted</th>
            </tr>
        </thead>
        <tbody>
            <?php if ($error_message): ?>
                <tr><td colspan="7" class="no-data-message"><?php echo htmlspecialchars($error_message); ?></td></tr>
            <?php elseif (!empty($allTransactions)): ?>
                <?php foreach ($allTransactions as $txn):
                    // Universal variables
                    $amount = (float)($txn->TotalAmt ?? 0);
                    $fullEntityShortName = (new ReflectionClass($txn))->getShortName();
                    $entityShortName = str_replace('IPP', '', $fullEntityShortName);
                    $theLine = (isset($txn->Line) && !empty($txn->Line)) ? (is_array($txn->Line) ? $txn->Line[0] : $txn->Line) : null;

                    // Defaults
                    $isExpense = true;
                    $fromTo = 'N/A';
                    $accountId = 'unknown';
                    
                    // From/To Logic
                    switch ($entityShortName) {
                        case 'Purchase': case 'Check':
                            $isExpense = true;
                            if (isset($txn->EntityRef)) { $entityId = (string) $txn->EntityRef; if (isset($entityNameMap[$entityId])) { $fromTo = htmlspecialchars($entityNameMap[$entityId]); } }
                            if (isset($txn->AccountRef)) { $accountId = (string) $txn->AccountRef; }
                            break;
                        case 'BillPayment':
                            $isExpense = true;
                            if (isset($txn->VendorRef)) { $vendorId = (string) $txn->VendorRef; if (isset($entityNameMap[$vendorId])) { $fromTo = htmlspecialchars($entityNameMap[$vendorId]); } }
                            if (isset($txn->CheckPayment->BankAccountRef)) { $accountId = (string) $txn->CheckPayment->BankAccountRef; } elseif (isset($txn->CreditCardPayment->CreditCardAccountRef)) { $accountId = (string) $txn->CreditCardPayment->CreditCardAccountRef; }
                            break;
                        case 'Payment': case 'SalesReceipt':
                            $isExpense = false;
                            if (isset($txn->CustomerRef)) { $customerId = (string) $txn->CustomerRef; if (isset($entityNameMap[$customerId])) { $fromTo = htmlspecialchars($entityNameMap[$customerId]); } }
                            if (isset($txn->DepositToAccountRef)) { $accountId = (string) $txn->DepositToAccountRef; }
                            break;
                        
                        case 'Deposit':
                            $isExpense = false;
                            $fromTo = 'Various'; // Default for multi-line deposits
                            if (isset($txn->DepositToAccountRef)) { $accountId = (string) $txn->DepositToAccountRef; }
                            
                            // --- FINAL FIX based on logs ---
                            if (isset($txn->Line)) {
                                $isSingleLine = is_array($txn->Line) ? (count($txn->Line) === 1) : true;
                                // For single-line deposits, check the Entity on the line detail
                                if ($isSingleLine && $theLine && isset($theLine->DepositLineDetail->Entity)) {
                                     // Cast the Entity object to a string to get its ID
                                     $entityId = (string) $theLine->DepositLineDetail->Entity;
                                     // Look up the ID in our combined Vendor/Customer map
                                     if(isset($entityNameMap[$entityId])) {
                                         $fromTo = htmlspecialchars($entityNameMap[$entityId]);
                                     }
                                }
                            }
                            break;
                    }
                    
                    // Full Description Logic
                    $description = '';
                    if (isset($txn->PrivateNote) && !empty(trim($txn->PrivateNote))) { $description = $txn->PrivateNote; }
                    if ($theLine) {
                        if (isset($theLine->Description) && !empty(trim($theLine->Description))) { $description = $theLine->Description; }
                        if (empty(trim($description)) && ($entityShortName === 'Payment' || $entityShortName === 'BillPayment') && isset($theLine->LinkedTxn)) {
                            $linkedTxn = is_array($theLine->LinkedTxn) ? $theLine->LinkedTxn[0] : $theLine->LinkedTxn;
                            $linkedType = str_replace('IPP', '', $linkedTxn->TxnType);
                            if (isset($linkedTxn->TxnId)) { $description = "Payment for {$linkedType} #" . $linkedTxn->TxnId; }
                        }
                    }

                    // Match/Category Logic
                    $matchCategoryDisplay = 'N/A';
                    if ($entityShortName === 'Payment' || $entityShortName === 'BillPayment' || (isset($txn->LinkedTxn) && !empty($txn->LinkedTxn))) {
                        $matchCategoryDisplay = 'Matched';
                    } elseif ($theLine) {
                        $categoryId = null;
                        if (isset($theLine->AccountBasedExpenseLineDetail->AccountRef)) { $categoryId = (string) $theLine->AccountBasedExpenseLineDetail->AccountRef; } 
                        elseif (isset($theLine->DepositLineDetail->AccountRef)) { $categoryId = (string) $theLine->DepositLineDetail->AccountRef; }
                        if ($categoryId && isset($accountNameMap[$categoryId])) { $matchCategoryDisplay = htmlspecialchars($accountNameMap[$categoryId]); }
                    }
                ?>
                <tr class="transaction-row" data-account-id="<?php echo htmlspecialchars($accountId); ?>">
                    <td><?php echo htmlspecialchars($txn->TxnDate ?? 'N/A'); ?></td>
                    <td><?php echo htmlspecialchars($description); ?></td>
                    <td class="currency"><?php echo ($isExpense) ? htmlspecialchars(number_format($amount, 2)) : ''; ?></td>
                    <td class="currency"><?php echo (!$isExpense) ? htmlspecialchars(number_format($amount, 2)) : ''; ?></td>
                    <td><?php echo $fromTo; ?></td>
                    <td><?php echo $matchCategoryDisplay; ?></td>
                    <td>
                        <a href="<?php echo getQBOTransactionLink($txn->Id, $entityShortName, $qboBaseConfig['baseUrl']); ?>" target="_blank" title="View in QuickBooks">
                           <?php echo $entityShortName; ?>
                        </a>
                    </td>
                </tr>
                <?php endforeach; ?>
            <?php else: ?>
                <tr><td colspan="7" class="no-data-message">No transactions found for the selected date range.</td></tr>
            <?php endif; ?>
        </tbody>
    </table>
</section>

<!-- Report Generation Timestamp -->
<div class="small-text">
    Banking data generated on <?php echo date('l j F Y H:i A T'); ?>
</div>