<?php
// qbo_vendor_list_test.php
// A test script to verify the advanced vendor matching logic.

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// --- Essential Includes ---
// We need the main QBO functions to connect and get the list
require_once __DIR__ . '/qbo_functions.php';
// NEW: We also need the file with our new matching functions
require_once __DIR__ . '/ai_email_functions.php';

require_once __DIR__ . '/../vendor/autoload.php';

use QuickBooksOnline\API\Core\OAuth\OAuth2\OAuth2LoginHelper;

// --- Configuration Paths ---
$qboBaseConfigFile = __DIR__ . '/config/qbo_config.php';
$qboTokenStorageFile = __DIR__ . '/tokens/qbo_token.json';

// --- Initialization ---
$qboUtil = null;
$qboErrorMessage = '';
$qboBaseConfig = null;
$vendorSearchMap = [];
$testResults = [];

// --- QBO Token Management & Initialization (Copied for standalone operation) ---
if (file_exists($qboBaseConfigFile)) { require_once $qboBaseConfigFile; } else { $qboErrorMessage = "CRITICAL: QBO base config file not found."; }
function loadQBOTokens(): ?array { global $qboTokenStorageFile; if (file_exists($qboTokenStorageFile)) { $json = file_get_contents($qboTokenStorageFile); if ($json === false) { return null; } return json_decode($json, true); } return null; }
function saveQBOTokens(string $accessToken, string $refreshToken): bool { global $qboTokenStorageFile; $tokens = ['access_token' => $accessToken, 'refresh_token' => $refreshToken, 'last_updated' => date('Y-m-d H:i:s')]; $tokenDir = dirname($qboTokenStorageFile); if (!is_dir($tokenDir)) { if (!mkdir($tokenDir, 0755, true) && !is_dir($tokenDir)) { return false; }} return file_put_contents($qboTokenStorageFile, json_encode($tokens, JSON_PRETTY_PRINT)); }
if (!$qboErrorMessage && $qboBaseConfig) {
    $currentTokens = loadQBOTokens(); $qboConfigForLibrary = $qboBaseConfig;
    if ($currentTokens && !empty($currentTokens['refresh_token'])) {
        try {
            $oauth2LoginHelper = new OAuth2LoginHelper($qboBaseConfig['ClientID'], $qboBaseConfig['ClientSecret']);
            $refreshedAccessTokenObj = $oauth2LoginHelper->refreshAccessTokenWithRefreshToken($currentTokens['refresh_token']);
            $newAccessToken = $refreshedAccessTokenObj->getAccessToken(); $newRefreshToken = $refreshedAccessTokenObj->getRefreshToken();
            if (saveQBOTokens($newAccessToken, $newRefreshToken)) { $currentTokens['access_token'] = $newAccessToken; }
        } catch (Exception $e) { $qboErrorMessage = "ERROR refreshing QBO tokens: " . $e->getMessage(); }
    }
    if (empty($qboErrorMessage) && $currentTokens && !empty($currentTokens['access_token'])) {
        $qboConfigForLibrary['accessTokenKey'] = $currentTokens['access_token']; $qboConfigForLibrary['refreshTokenKey'] = $currentTokens['refresh_token'];
        try { $qboUtil = new QBOUtilityLibrary($qboConfigForLibrary); } catch (Exception $e) { $qboErrorMessage = "Error instantiating QBOUtilityLibrary: " . $e->getMessage(); }
    } elseif (empty($qboErrorMessage)) { $qboErrorMessage = "Could not obtain a valid QBO access token."; }
}

/**
 * NEW HELPER FUNCTION
 * Generates an array of cleaned search variations from a single name.
 * This will be used by both the map builder and the matching function.
 *
 * @param string $name The input vendor name.
 * @return array An array of cleaned search terms.
 */
function generateSearchVariations(string $name): array
{
    // Terms to strip out to reduce noise (case-insensitive)
    $noise = [' pty ltd', ' pty', ' ltd', ' inc', ' llc', ' group', ' hotel'];
    $searchTerms = [];

    // 1. Add the original name as a baseline
    $searchTerms[] = trim($name);

    // 2. Check for a "Trading As" name (T/A)
    if (preg_match('/t\s?\/?\s?a\s(.+)/i', $name, $matches)) {
        array_unshift($searchTerms, trim($matches[1])); // Add with high priority
    }
    
    // 3. Add a version with common legal/generic noise removed
    $cleanName = str_ireplace($noise, '', $name);
    if (trim($cleanName) !== trim($name)) {
        $searchTerms[] = trim($cleanName);
    }
    
    // 4. Add a version with content in parentheses removed
    $noParenthesesName = trim(preg_replace('/\s*\(.*?\)\s*/', ' ', $name));
    if (trim($noParenthesesName) !== trim($name)) {
        $searchTerms[] = trim($noParenthesesName);
    }
    
    // Return only the unique variations
    return array_unique($searchTerms);
}

/**
 * MODIFIED FUNCTION
 * Builds a search-optimized map by calling the new helper function.
 *
 * @param array $vendors The raw array of vendor objects from the QBO API.
 * @return array The optimized search map.
 */
function buildVendorSearchMap(array $vendors): array
{
    $searchMap = [];
    foreach ($vendors as $vendor) {
        if (!empty($vendor->DisplayName)) {
            // Use the new helper to generate variations for each vendor
            $searchMap[$vendor->DisplayName] = generateSearchVariations($vendor->DisplayName);
        }
    }
    return $searchMap;
}

/**
 * MODIFIED FUNCTION
 * Finds the best vendor match using the new symmetrical normalization approach.
 *
 * @param string $targetName The name from the AI/invoice.
 * @param array $vendorSearchMap The map created by buildVendorSearchMap().
 * @return string|null The best matching original vendor name.
 */
function findBestVendorMatch(string $targetName, array $vendorSearchMap): ?string
{
    $bestMatch = null;
    $highestPercentage = 0.0;

    // 1. Generate the same clean variations for the user's input name
    $targetVariations = generateSearchVariations($targetName);

    // 2. Loop through the main map (Original Name => [db_search_term_1, ...])
    foreach ($vendorSearchMap as $originalName => $dbSearchTerms) {
        // 3. Perform a "matrix" comparison: check every input variation against every DB variation
        foreach ($targetVariations as $targetTerm) {
            foreach ($dbSearchTerms as $dbTerm) {
                similar_text(strtolower($targetTerm), strtolower($dbTerm), $percent);

                if ($percent > 99) { // An almost perfect match is good enough
                    return $originalName; // Return immediately
                }

                if ($percent > $highestPercentage) {
                    $highestPercentage = $percent;
                    $bestMatch = $originalName;
                }
            }
        }
    }

    // 4. Use a threshold to ensure the best match found is still a good one.
    // 80% is a good starting point for this more robust comparison.
    if ($highestPercentage > 80) {
        return $bestMatch;
    }

    return null;
}

// --- Main Test Logic ---
if ($qboUtil) {
    try {
        // 1. Fetch all vendors from QBO
        $allVendors = [];
        $startPosition = 1;
        while (true) {
            $vendorsBatch = $qboUtil->getDataService()->Query("SELECT * FROM Vendor WHERE Active = true STARTPOSITION {$startPosition} MAXRESULTS 1000");
            if (empty($vendorsBatch)) { break; }
            $allVendors = array_merge($allVendors, $vendorsBatch);
            $startPosition += count($vendorsBatch);
        }

        // 2. Build the optimized search map
        // Ensure the function exists before calling it
        if (function_exists('buildVendorSearchMap')) {
            $vendorSearchMap = buildVendorSearchMap($allVendors);
        } else {
            throw new Exception("Function 'buildVendorSearchMap' not found in included files.");
        }

        // 3. Define our test cases
        $testCases = [
            'Ibis Styles Sydney Central', // The key test case
            'ibis sydney central',        // Test case-insensitivity
            'Fantasea Cruising Sydney',   // Test variation
            'AAT Kings',                  // Test exact match
            'BridgeClimb',                // Test slight variation of 'Bridge Climb Sydney'
            'Aon Insurance',              // Test name with a common word
            'NonExistent Vendor 123'      // Test a name that shouldn't match anything
        ];

        // 4. Run the tests
        if (function_exists('findBestVendorMatch')) {
            foreach ($testCases as $testName) {
                $matchedName = findBestVendorMatch($testName, $vendorSearchMap);
                $testResults[] = [
                    'input_name' => $testName,
                    'matched_name' => $matchedName ?? 'No Match Found'
                ];
            }
        } else {
            throw new Exception("Function 'findBestVendorMatch' not found in included files.");
        }
    } catch (Exception $e) {
        $qboErrorMessage = "An error occurred during testing: " . $e->getMessage();
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Advanced Vendor Matching Test</title>
    <style>
        body { font-family: sans-serif; line-height: 1.6; margin: 20px; background-color: #f4f7f6; }
        .container { max-width: 900px; margin: 0 auto; padding: 20px; background-color: #fff; border-radius: 8px; box-shadow: 0 0 15px rgba(0,0,0,0.1); }
        h1, h2 { color: #005ea2; border-bottom: 2px solid #eee; padding-bottom: 10px; }
        .message { padding: 15px; margin: 20px 0; border-radius: 4px; border: 1px solid transparent; }
        .error { background-color: #f8d7da; color: #721c24; border-color: #f5c6cb; }
        .success { background-color: #d4edda; color: #155724; border-color: #c3e6cb; }
        table { width: 100%; border-collapse: collapse; margin-top: 20px; }
        th, td { padding: 12px; border: 1px solid #ddd; text-align: left; }
        th { background-color: #e9ecef; }
        tr:nth-child(even) { background-color: #f8f9fa; }
        code { background-color: #e9ecef; padding: 2px 4px; border-radius: 3px; font-family: monospace; }
    </style>
</head>
<body>
    <div class="container">
        <h1>Advanced Vendor Matching Test</h1>
        <h2>Live Match Test</h2>
        <div style="background-color: #f8f9fa; padding: 15px; border-radius: 5px; border: 1px solid #dee2e6; margin-bottom: 30px;">
            <form method="POST" action="">
                <label for="user_test_name" style="font-weight: bold; display: block; margin-bottom: 8px;">Enter a Vendor Name to Test:</label>
                <input type="text" id="user_test_name" name="user_test_name" style="width: 70%; padding: 8px; border: 1px solid #ced4da; border-radius: 4px;" placeholder="e.g., ibis sydney central">
                <button type="submit" style="padding: 8px 15px; border: none; background-color: #007bff; color: white; border-radius: 4px; cursor: pointer;">Find Best Match</button>
            </form>
        
            <?php
            // Logic to handle the form submission
            if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($_POST['user_test_name'])) {
                $userTestName = htmlspecialchars($_POST['user_test_name']);
                $userMatchedName = 'Error: Search map not built.';
                if (!empty($vendorSearchMap)) {
                    $userMatchedName = findBestVendorMatch($userTestName, $vendorSearchMap) ?? 'No Match Found';
                }
                
                // Display the result of the user's test
                echo '<div style="margin-top: 15px; padding: 10px; background-color: #e2e3e5; border-radius: 4px;">';
                echo '<strong>Test Result for "'. $userTestName .'":</strong> ';
                echo '<span>'. htmlspecialchars($userMatchedName) .'</span>';
                echo '</div>';
            }
            ?>
        </div>

        <?php if (!empty($qboErrorMessage)): ?>
            <div class="message error">
                <strong>An error occurred:</strong><br>
                <?php echo htmlspecialchars($qboErrorMessage); ?>
            </div>
        <?php else: ?>
            <div class="message success">
                Successfully fetched <?php echo count($vendorSearchMap); ?> vendors and built the search map. Running tests...
            </div>

            <h2>Test Results</h2>
            <table>
                <thead>
                    <tr>
                        <th>Input Name (from AI)</th>
                        <th>Best Matched QBO Vendor Name</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($testResults as $result): ?>
                        <tr>
                            <td><code><?php echo htmlspecialchars($result['input_name']); ?></code></td>
                            <td><?php echo htmlspecialchars($result['matched_name']); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>

            <h2>How it Works</h2>
            <p>This test proves that the matching logic can handle complex cases:</p>
            <ul>
                <li>It extracts "trading as" names (like finding <code>Ibis Styles Sydney Central</code> inside a long legal name).</li>
                <li>It handles case differences and common variations.</li>
                <li>It performs these lookups instantly using PHP, without any extra API calls or tokens.</li>
            </ul>

        <?php endif; ?>
    </div>
</body>
</html>