<?php
// test_spam_classification.php

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Required files
require_once "dbconn.php"; // For database connection ($conn)
require_once "ai_keys.php"; // For apiURL_Flash()
require_once "file_extract_functions.php"; // For attachment text extraction
require_once "ai_email_functions.php"; // The main AI functions file

$is_cli = (php_sapi_name() === 'cli');

// --- Database Connection ---
if (!$conn) {
    if ($is_cli) {
        die("Failed to establish database connection. Check dbconn.php.\n");
    } else {
        echo "<!DOCTYPE html><html><head><title>Spam Classification Test Error</title></head><body>";
        echo "<h1>Spam Classification Test</h1>";
        echo "<p style='color:red;'><strong>Error:</strong> Failed to establish database connection. Check dbconn.php.</p>";
        echo "</body></html>";
        exit;
    }
}
if ($conn->connect_error) {
     if ($is_cli) {
        die("Database Connection Error: " . $conn->connect_error . "\n");
    } else {
        echo "<!DOCTYPE html><html><head><title>Spam Classification Test Error</title></head><body>";
        echo "<h1>Spam Classification Test</h1>";
        echo "<p style='color:red;'><strong>Database Connection Error:</strong> " . htmlspecialchars($conn->connect_error) . "</p>";
        echo "</body></html>";
        exit;
    }
}

// --- HTML Header for Web View ---
if (!$is_cli) {
    echo "<!DOCTYPE html>";
    echo "<html lang='en'>";
    echo "<head>";
    echo "<meta charset='UTF-8'>";
    echo "<meta name='viewport' content='width=device-width, initial-scale=1.0'>";
    echo "<title>Spam Classification Test</title>";
    echo "<style>";
    echo "body { font-family: sans-serif; margin: 20px; background-color: #f4f4f4; color: #333; }";
    echo "h1, h2 { color: #0056b3; }";
    echo ".container { background-color: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 0 10px rgba(0,0,0,0.1); }";
    echo "form { margin-bottom: 20px; }";
    echo "label { display: block; margin-bottom: 5px; }";
    echo "input[type='text'] { width: 300px; padding: 8px; margin-bottom: 10px; border: 1px solid #ddd; border-radius: 4px; }";
    echo "input[type='submit'] { padding: 10px 15px; background-color: #0056b3; color: white; border: none; border-radius: 4px; cursor: pointer; }";
    echo "input[type='submit']:hover { background-color: #004494; }";
    echo "pre { background-color: #e9e9e9; padding: 15px; border-radius: 4px; white-space: pre-wrap; word-wrap: break-word; font-size: 0.9em; max-height: 400px; overflow-y: auto; border: 1px solid #ccc;}";
    echo ".result { margin-top: 20px; padding: 15px; border-radius: 4px; }";
    echo ".result.yes { background-color: #ffe0e0; border: 1px solid #ffb3b3; color: #c00; }"; // Light red for spam
    echo ".result.no { background-color: #e0ffe0; border: 1px solid #b3ffb3; color: #008000; }"; // Light green for not spam
    echo ".result.error { background-color: #fff0b3; border: 1px solid #ffe580; color: #997400; }"; // Light yellow for error
    echo "</style>";
    echo "</head>";
    echo "<body>";
    echo "<div class='container'>";
    echo "<h1>Spam Classification Test</h1>";
}

// --- User Input for Conversation ID ---
$conversation_id = null;
$form_submitted = false;

if ($is_cli) {
    $conversation_id = readline("Enter Conversation ID: ");
} else {
    if (isset($_REQUEST['conversation_id'])) {
        $conversation_id = trim($_REQUEST['conversation_id']);
        $form_submitted = true;
    }
    // Display form
    echo "<form method='POST'>";
    echo "<label for='conversation_id'>Enter Conversation ID:</label>";
    echo "<input type='text' name='conversation_id' id='conversation_id' value=\"" . htmlspecialchars($conversation_id ?? '') . "\">";
    echo "<input type='submit' value='Classify Spam'>";
    echo "</form>";
}

if (($is_cli && empty($conversation_id)) || (!$is_cli && !$form_submitted)) {
    if ($is_cli) {
        echo "No Conversation ID provided.\n";
    }
    // For web, the form is displayed, so we can just stop further PHP execution if no ID submitted yet.
    if (!$is_cli && !$form_submitted) {
        echo "</div></body></html>"; // Close HTML tags
        if ($conn) $conn->close();
        exit;
    }
     if (!$is_cli && empty($conversation_id) && $form_submitted) {
        echo "<p style='color:orange;'>Please enter a Conversation ID.</p>";
        echo "</div></body></html>"; // Close HTML tags
        if ($conn) $conn->close();
        exit;
    }
}
if (empty($conversation_id) && $form_submitted && !$is_cli){ // Edge case if submitted empty
     echo "<p style='color:orange;'>Please enter a Conversation ID.</p>";
} elseif (!empty($conversation_id)) {

    if ($is_cli) {
        echo "Processing Conversation ID: " . htmlspecialchars($conversation_id) . "\n\n";
    } else {
        echo "<h2>Processing Results for Conversation ID: " . htmlspecialchars($conversation_id) . "</h2>";
    }

    // --- Reconstruct and Display the Prompt ---
    if ($is_cli) {
        echo "-------------------------------------\n";
        echo "RECONSTRUCTED PROMPT FOR SPAM AI:\n";
        echo "-------------------------------------\n";
    } else {
        echo "<h3>Reconstructed Prompt for Spam AI:</h3>";
    }

    $prompt_for_display = "";
    $conversationID_safe_for_prompt = mysqli_real_escape_string($conn, $conversation_id);
    $combinedContentForAI_for_prompt = "";

    $allConversationData_for_prompt = getAllEmailMessagesAndAttachmentsInConversation($conversationID_safe_for_prompt, $conn);

    if (!empty($allConversationData_for_prompt['emails'])) {
        foreach ($allConversationData_for_prompt['emails'] as $email) {
            $combinedContentForAI_for_prompt .= "\n\n--- Email Start (Sender: " . ($email['sender'] ?? 'N/A') . ", Received: " . ($email['received_datetime'] ?? 'N/A') . ") ---\n";
            $combinedContentForAI_for_prompt .= "Subject: " . ($email['subject'] ?? '') . "\n";
            $combinedContentForAI_for_prompt .= "Body:\n" . ($email['body_clean'] ?? '');
            $combinedContentForAI_for_prompt .= "\n--- Email End ---";
        }
    }
    if (!empty($allConversationData_for_prompt['attachments'])) {
        foreach ($allConversationData_for_prompt['attachments'] as $attachment) {
            if (!$attachment['error'] && !empty(trim($attachment['extracted_text']))) {
                 $combinedContentForAI_for_prompt .= "\n\n--- Attachment: " . htmlspecialchars($attachment['filename'] ?? 'N/A') . " (From email received: " . ($attachment['parent_email_received_datetime'] ?? 'N/A') . ") ---\n" . trim($attachment['extracted_text']);
                 $combinedContentForAI_for_prompt .= "\n--- Attachment End ---";
            }
        }
    }
    $combinedContentForAI_for_prompt = trim($combinedContentForAI_for_prompt);

    if (empty($combinedContentForAI_for_prompt) && (empty($allConversationData_for_prompt['emails']) || empty(end($allConversationData_for_prompt['emails'])['subject']))) {
        $prompt_text_content = get_ai_spam_classification_prompt_text() . "\n[No textual content found in conversation for prompt display]";
    } else {
        $prompt_text_content = get_ai_spam_classification_prompt_text() . "\n" . $combinedContentForAI_for_prompt;
    }

    if ($is_cli) {
        // UTF-8 check for display (primarily for CLI)
        if (!mb_check_encoding($prompt_text_content, 'UTF-8')) {
            $detected_encoding = mb_detect_encoding($prompt_text_content, 'UTF-8, Windows-1252, ISO-8859-1', true);
            $converted_prompt_for_display = mb_convert_encoding($prompt_text_content, 'UTF-8', $detected_encoding ?: 'UTF-8');
            echo $converted_prompt_for_display ?: "[Error converting prompt to UTF-8 for display]\n" . $prompt_text_content;
        } else {
            echo $prompt_text_content;
        }
        echo "\n-------------------------------------\n\n";
    } else {
        echo "<pre>" . htmlspecialchars($prompt_text_content) . "</pre>";
    }


    // --- Call classifyAsSpam function ---
    if ($is_cli) {
        echo "-------------------------------------\n";
        echo "SPAM CLASSIFICATION RESULT:\n";
        echo "-------------------------------------\n";
    } else {
        echo "<h3>Spam Classification Result:</h3>";
    }

    $spam_classification_result = classifyAsSpam($conversation_id, $conn);
    $result_message = "";
    $result_class = "";

    if ($spam_classification_result === 'yes') {
        $result_message = "The conversation (ID: " . htmlspecialchars($conversation_id) . ") is classified as: <strong>SPAM / Auto-Reply (yes)</strong>";
        $result_class = "yes";
    } elseif ($spam_classification_result === 'no') {
        $result_message = "The conversation (ID: " . htmlspecialchars($conversation_id) . ") is classified as: <strong>NOT SPAM / Needs Attention (no)</strong>";
        $result_class = "no";
    } elseif ($spam_classification_result === 'error') {
        $result_message = "An <strong>error</strong> occurred during spam classification for Conversation ID: " . htmlspecialchars($conversation_id) . ". Check server logs (email_ai_functions_error.log).";
        $result_class = "error";
    } else {
        $result_message = "Received an unexpected result for spam classification: '<strong>" . htmlspecialchars($spam_classification_result) . "</strong>' for Conversation ID: " . htmlspecialchars($conversation_id);
        $result_class = "error";
    }

    if ($is_cli) {
        echo strip_tags($result_message) . "\n"; // Remove HTML tags for CLI
        echo "-------------------------------------\n";
    } else {
        echo "<div class='result " . $result_class . "'>" . $result_message . "</div>";
    }
}

// --- HTML Footer for Web View ---
if (!$is_cli) {
    echo "</div>"; // Close container
    echo "</body>";
    echo "</html>";
}

// Close the database connection
if ($conn) {
    $conn->close();
}
?>