<?php
// test_tag_classification.php

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Required files
require_once "dbconn.php"; // For database connection ($conn)
require_once "ai_keys.php"; // For apiURL_Flash()
require_once "file_extract_functions.php"; // For attachment text extraction
require_once "ai_email_functions.php"; // The main AI functions file

$is_cli = (php_sapi_name() === 'cli');

// --- Database Connection ---
if (!$conn) {
    $error_message = "Failed to establish database connection. Check dbconn.php.";
    if ($is_cli) die($error_message . "\n");
    echo "<!DOCTYPE html><html><head><title>Tag Classification Test Error</title></head><body><h1>Tag Classification Test</h1><p style='color:red;'><strong>Error:</strong> $error_message</p></body></html>";
    exit;
}
if ($conn->connect_error) {
    $error_message = "Database Connection Error: " . htmlspecialchars($conn->connect_error);
    if ($is_cli) die(strip_tags($error_message) . "\n");
    echo "<!DOCTYPE html><html><head><title>Tag Classification Test Error</title></head><body><h1>Tag Classification Test</h1><p style='color:red;'><strong>$error_message</strong></p></body></html>";
    exit;
}

// --- HTML Header for Web View ---
if (!$is_cli) {
    echo "<!DOCTYPE html><html lang='en'><head><meta charset='UTF-8'><meta name='viewport' content='width=device-width, initial-scale=1.0'><title>Tag Classification Test</title><style>";
    echo "body { font-family: sans-serif; margin: 20px; background-color: #f4f4f4; color: #333; }";
    echo "h1, h2, h3 { color: #0056b3; }";
    echo ".container { background-color: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 0 10px rgba(0,0,0,0.1); }";
    echo "form { margin-bottom: 20px; } label { display: block; margin-bottom: 5px; }";
    echo "input[type='text'] { width: 300px; padding: 8px; margin-bottom: 10px; border: 1px solid #ddd; border-radius: 4px; }";
    echo "input[type='submit'] { padding: 10px 15px; background-color: #0056b3; color: white; border: none; border-radius: 4px; cursor: pointer; }";
    echo "input[type='submit']:hover { background-color: #004494; }";
    echo "pre { background-color: #e9e9e9; padding: 15px; border-radius: 4px; white-space: pre-wrap; word-wrap: break-word; font-size: 0.9em; max-height: 500px; overflow-y: auto; border: 1px solid #ccc;}";
    echo ".result { margin-top: 20px; padding: 15px; border-radius: 4px; }";
    echo ".result.success { background-color: #e0ffe0; border: 1px solid #b3ffb3; color: #008000; }";
    echo ".result.nodata { background-color: #e6f7ff; border: 1px solid #ade6ff; color: #006080; }";
    echo ".result.error { background-color: #fff0b3; border: 1px solid #ffe580; color: #997400; }";
    echo "ul { list-style-type: square; padding-left: 20px; } li { margin-bottom: 5px; }";
    echo "</style></head><body><div class='container'><h1>Tag Classification Test</h1>";
}

// --- User Input for Conversation ID ---
$conversation_id = null;
$form_submitted = false;
// Define excluded categories for tags (can be modified as needed)
$excludedCategoriesForTags = []; // Example: ['internal_notes', 'system_generated']

if ($is_cli) {
    $conversation_id = readline("Enter Conversation ID: ");
    $excluded_input = readline("Enter comma-separated categories to EXCLUDE for tags (leave blank for none): ");
    if (!empty(trim($excluded_input))) {
        $excludedCategoriesForTags = array_map('trim', explode(',', $excluded_input));
    }
} else {
    if (isset($_REQUEST['conversation_id'])) {
        $conversation_id = trim($_REQUEST['conversation_id']);
        if (isset($_REQUEST['excluded_categories'])) {
            $excluded_raw = trim($_REQUEST['excluded_categories']);
            if (!empty($excluded_raw)) {
                 $excludedCategoriesForTags = array_map('trim', explode(',', $excluded_raw));
            }
        }
        $form_submitted = true;
    }
    echo "<form method='POST'>";
    echo "<label for='conversation_id'>Enter Conversation ID:</label>";
    echo "<input type='text' name='conversation_id' id='conversation_id' value=\"" . htmlspecialchars($conversation_id ?? '') . "\">";
    echo "<label for='excluded_categories'>Exclude Tag Categories (comma-separated, optional):</label>";
    echo "<input type='text' name='excluded_categories' id='excluded_categories' placeholder='e.g. internal, system' value=\"" . htmlspecialchars(implode(', ', $excludedCategoriesForTags)) . "\">";
    echo "<input type='submit' value='Suggest Tags'>";
    echo "</form>";
}

if (($is_cli && empty($conversation_id)) || (!$is_cli && !$form_submitted)) {
    if ($is_cli) echo "No Conversation ID provided.\n";
    if (!$is_cli && !$form_submitted) { echo "</div></body></html>"; if ($conn) $conn->close(); exit; }
    if (!$is_cli && empty($conversation_id) && $form_submitted) {
        echo "<p style='color:orange;'>Please enter a Conversation ID.</p>";
        echo "</div></body></html>"; if ($conn) $conn->close(); exit;
    }
}
if (empty($conversation_id) && $form_submitted && !$is_cli){
     echo "<p style='color:orange;'>Please enter a Conversation ID.</p>";
} elseif (!empty($conversation_id)) {

    if ($is_cli) {
        echo "Processing Conversation ID: " . htmlspecialchars($conversation_id) . "\n";
        if (!empty($excludedCategoriesForTags)) {
            echo "Excluding Tag Categories: " . htmlspecialchars(implode(', ', $excludedCategoriesForTags)) . "\n\n";
        }
    } else {
        echo "<h2>Processing Results for Conversation ID: " . htmlspecialchars($conversation_id) . "</h2>";
         if (!empty($excludedCategoriesForTags)) {
            echo "<p><strong>Excluding Tag Categories:</strong> " . htmlspecialchars(implode(', ', $excludedCategoriesForTags)) . "</p>";
        }
    }

    // --- Reconstruct and Display the Prompt for Tag Suggestion AI ---
    if ($is_cli) {
        echo "-------------------------------------\n";
        echo "RECONSTRUCTED PROMPT FOR TAG SUGGESTION AI:\n";
        echo "-------------------------------------\n";
    } else {
        echo "<h3>Reconstructed Prompt for Tag Suggestion AI:</h3>";
    }

    $prompt_for_display = "";
    $conversationContentForAI_for_prompt = "";

    // Get content for AI (latest email subject/body + all attachments text)
    $latestEmailData_for_prompt = getMessageFromConversationID_forSegments($conversation_id, $conn);
    if ($latestEmailData_for_prompt) {
        $emailSubject = $latestEmailData_for_prompt['subject'] ?? '';
        $emailBodyRaw = $latestEmailData_for_prompt['plaintext_body'] ?? ($latestEmailData_for_prompt['full_body'] ?? '');
        $emailBodyClean = strip_tags($emailBodyRaw);
        $conversationContentForAI_for_prompt .= "Email Subject: " . trim($emailSubject) . "\nEmail Body:\n" . trim($emailBodyClean) . "\n\n";
    }

    $attachmentItems_for_prompt = getAttachmentFiles($conversation_id, $conn);
    if (!empty($attachmentItems_for_prompt)) {
        $conversationContentForAI_for_prompt .= "--- Attachments Text ---\n";
        foreach ($attachmentItems_for_prompt as $item) {
            $conversationContentForAI_for_prompt .= "\n--- Attachment: " . htmlspecialchars($item['filename']) . " ---\n" . $item['text'] . "\n";
        }
    }
    $trimmedCombinedContent_for_prompt = trim($conversationContentForAI_for_prompt);
    if (empty($trimmedCombinedContent_for_prompt)) {
        $prompt_text_content = "[No textual content found in conversation for prompt display]";
    } else {
        // Get all available tags from DB
        $allTagsGrouped_for_prompt = getAllTags($conn);
        $tagListForPrompt_for_display = "";
        if (empty($allTagsGrouped_for_prompt)) {
             $tagListForPrompt_for_display = "No tags found in the database.\n";
        } else {
            foreach ($allTagsGrouped_for_prompt as $category => $tags) {
                if (in_array($category, $excludedCategoriesForTags)) continue;
                $tagListForPrompt_for_display .= "Category '$category':\n";
                foreach ($tags as $tag) {
                    $tagListForPrompt_for_display .= "- '" . addslashes($tag['tag']) . "' (ID: " . $tag['id'] . ")\n";
                }
                $tagListForPrompt_for_display .= "\n";
            }
        }
        if(empty(trim($tagListForPrompt_for_display))) {
            $tagListForPrompt_for_display = "No tags available for AI after applying exclusions.\n";
        }

        // Replicate $currentAvailableSystemTypes and $typeUserAssignments logic for accurate prompt display
        $currentAvailableSystemTypes_for_prompt = [
            110 => 'Complaint', 92  => 'Confirmation', 108 => 'Following up Vouchers',
            109 => 'Payment on Arrival', 88  => 'Post Sales', 101 => 'Post Sales In Progress',
            89  => 'Pre Sales', 90  => 'Quotes', 105 => 'Refund', 91  => 'Requote',
            107 => 'Requote after Confirmation'
        ];
        $typeUserAssignments_for_prompt = getTypeToUserAssignments($conn, $currentAvailableSystemTypes_for_prompt);
        $personRulesString_for_display = "Rules for person assignment based on specific types:\n";
        if (!empty($typeUserAssignments_for_prompt)) {
            foreach ($typeUserAssignments_for_prompt as $type => $users) {
                $personRulesString_for_display .= "- If the conversation is related to '$type', assign person tag for: '$users'.\n";
            }
        } else {
            $personRulesString_for_display .= "- No specific dynamic person assignment rules found from database. Use general knowledge if applicable.\n";
        }

        $preprompt_for_display = "Analyze 'Conversation Content'.
Suggest relevant tags (by ID) from 'Available Tags List'.
Output ONLY a valid JSON array of strings (tag IDs).
Empty array [] if none. No other text/markdown. \n
Information: The North, West, East, South tags correspond to regions of India that the vendor is from (anyone not TDU), so look for indian regions or zipcode and assign tags accordingly, the latest one takes precedence.
Exempt 'Turtle Down Under' from the analysis for all tags.
If the assign person is blank, that means no one is assigned to it and can be safely ignored
$personRulesString_for_display
The following are considered types Post Sales, Pre Sales, Quotes, Requote, Confirmation, Post Sales In Progress, Refund, Requote after Confirmation, Following up Vouchers, Payment on Arrival, Complaint. There can only be 1 types, consider the latest situation within the conversation to determine the most appropriate type\nAvailable Tags List:\n$tagListForPrompt_for_display\nConversation Content:\n";
        $prompt_text_content = $preprompt_for_display . $trimmedCombinedContent_for_prompt;
    }


    if ($is_cli) {
        if (!mb_check_encoding($prompt_text_content, 'UTF-8')) {
            $detected_encoding = mb_detect_encoding($prompt_text_content, 'UTF-8, Windows-1252, ISO-8859-1', true);
            $converted_prompt_for_display = mb_convert_encoding($prompt_text_content, 'UTF-8', $detected_encoding ?: 'UTF-8');
            echo $converted_prompt_for_display ?: "[Error converting prompt to UTF-8 for display]\n" . $prompt_text_content;
        } else {
            echo $prompt_text_content;
        }
        echo "\n-------------------------------------\n\n";
    } else {
        echo "<pre>" . htmlspecialchars($prompt_text_content) . "</pre>";
    }

    // --- Call getSuggestedTagsForConversation function ---
    if ($is_cli) {
        echo "-------------------------------------\n";
        echo "SUGGESTED TAGS RESULT:\n";
        echo "-------------------------------------\n";
    } else {
        echo "<h3>Suggested Tags Result:</h3>";
    }

    $suggested_tags_result = getSuggestedTagsForConversation($conversation_id, $conn, $excludedCategoriesForTags);
    $result_message = "";
    $result_class = "";

    if ($suggested_tags_result === null || !is_array($suggested_tags_result)) { // null can happen on API error inside function
        $result_message = "An <strong>error</strong> occurred during tag suggestion for Conversation ID: " . htmlspecialchars($conversation_id) . ". Check server logs (email_ai_functions_error.log).";
        $result_class = "error";
    } elseif (empty($suggested_tags_result)) {
        $result_message = "No tags were suggested by the AI for Conversation ID: " . htmlspecialchars($conversation_id) . ".";
        $result_class = "nodata";
    } else {
        $result_message = "<strong>Suggested Tags for Conversation ID " . htmlspecialchars($conversation_id) . ":</strong><ul>";
        foreach ($suggested_tags_result as $tag) {
            $result_message .= "<li>ID: " . htmlspecialchars($tag['id']) . " - Name: " . htmlspecialchars($tag['name']) . "</li>";
        }
        $result_message .= "</ul>";
        $result_class = "success";
    }

    if ($is_cli) {
        echo strip_tags(str_replace(["<li>", "</li>", "<ul>", "</ul>"], ["\n- ", "", "\n", ""], $result_message)) . "\n";
        echo "-------------------------------------\n";
    } else {
        echo "<div class='result " . $result_class . "'>" . $result_message . "</div>";
    }
}

// --- HTML Footer for Web View ---
if (!$is_cli) {
    echo "</div></body></html>";
}

// Close the database connection
if ($conn) {
    $conn->close();
}
?>