<?php
// report_outreach_test.php
// A simple test report to list all available outreach activity data.

// Include your standard database connection file
require_once __DIR__ . '/dbconn.php'; // Adjust the path if necessary

// Check if the connection to the database was successful
if (!$conn) {
    die("CRITICAL ERROR: Database connection failed. Please check your dbconn.php file.");
}

$page_title = "Outreach Activity Test Report";

// Optional: Include a standard header if you have one
if (file_exists('header.php')) {
    include 'header.php';
}

// --- Main Data Fetching Logic ---
$all_followups = [];
$error_message = null;

try {
    // The main SQL query to get all the data.
    // We also JOIN with the tdu_organisation table to get the Organisation Name for context.
    $sql = "SELECT 
                f.created_by AS agent_user,
                f.visit_time,
                f.mode,
                f.created_at AS upload_date,
                f.next_follow_up_date,
                o.organization_name
            FROM 
                tdu_organisation_followup f
            LEFT JOIN 
                tdu_organisation o ON f.organizationid = o.organizationid
            ORDER BY 
                f.visit_time DESC"; // Order by the most recent visit time

    $result = $conn->query($sql);

    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $all_followups[] = $row;
        }
    } else {
        // This will catch errors in the SQL query itself
        throw new Exception("Database query failed: " . $conn->error);
    }

} catch (Exception $e) {
    // Catch any exceptions during the process
    $error_message = "An error occurred: " . $e->getMessage();
}

// Close the database connection
$conn->close();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title><?= htmlspecialchars($page_title) ?></title>
    <style>
        body { font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif; margin: 20px; background-color: #f4f6f9; }
        .report-container { background-color: #fff; border: 1px solid #dee2e6; border-radius: 8px; padding: 20px; max-width: 1200px; margin: auto; }
        h1 { text-align: center; color: #343a40; margin-bottom: 25px; }
        .styled-table { width: 100%; border-collapse: collapse; font-size: 14px; }
        .styled-table th, .styled-table td { border: 1px solid #ddd; padding: 12px; text-align: left; }
        .styled-table th { background-color: #f2f2f2; font-weight: bold; color: #495057; }
        .styled-table tbody tr:nth-of-type(even) { background-color: #f9f9f9; }
        .styled-table tbody tr:hover { background-color: #e9ecef; }
        .error-message { color: #d93025; font-weight: bold; text-align: center; padding: 20px; }
        .no-results { color: #6c757d; text-align: center; padding: 20px; }
    </style>
</head>
<body>

    <div class="report-container">
        <h1><?= htmlspecialchars($page_title) ?></h1>

        <?php if ($error_message): ?>
            <p class="error-message"><?= htmlspecialchars($error_message) ?></p>

        <?php elseif (empty($all_followups)): ?>
            <p class="no-results">No outreach activity data found in the database.</p>

        <?php else: ?>
            <table class="styled-table">
                <thead>
                    <tr>
                        <th>User (Agent)</th>
                        <th>Organisation Name</th>
                        <th>Visit Time</th>
                        <th>Mode</th>
                        <th>Upload Date</th>
                        <th>Next Follow-up Date</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($all_followups as $row): ?>
                        <tr>
                            <td><?= htmlspecialchars($row['agent_user']) ?></td>
                            <td><?= htmlspecialchars($row['organization_name']) ?></td>
                            <td><?= !empty($row['visit_time']) ? (new DateTime($row['visit_time']))->format('d-M-Y g:i A') : 'N/A' ?></td>
                            <td><?= htmlspecialchars($row['mode']) ?></td>
                            <td><?= !empty($row['upload_date']) ? (new DateTime($row['upload_date']))->format('d-M-Y g:i A') : 'N/A' ?></td>
                            <td><?= !empty($row['next_follow_up_date']) ? (new DateTime($row['next_follow_up_date']))->format('d-M-Y') : 'N/A' ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>

</body>
</html>