<?php
// report_outreach_final.php
// An interactive report for analyzing sales agent outreach activities.
// ENHANCED: Uses Flatpickr for compact date range filtering.

require_once __DIR__ . '/dbconn.php'; // Adjust path if necessary

if (!$conn) {
    die("CRITICAL ERROR: Database connection failed.");
}

$page_title = "Agent Outreach Activity Report";

// --- Data Fetching ---
$all_followups = [];
$error_message = null;
$agent_list = [];

try {
    $sql = "SELECT 
                f.created_by AS agent_user,
                f.visit_time,
                f.mode,
                f.created_at AS upload_date,
                f.next_follow_up_date,
                o.organization_name
            FROM 
                tdu_organisation_followup f
            LEFT JOIN 
                tdu_organisation o ON f.organizationid = o.organizationid
            ORDER BY 
                f.visit_time DESC";

    $result = $conn->query($sql);

    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $all_followups[] = $row;
            if (!in_array($row['agent_user'], $agent_list)) {
                $agent_list[] = $row['agent_user'];
            }
        }
        sort($agent_list);
    } else {
        throw new Exception("Database query failed: " . $conn->error);
    }

} catch (Exception $e) {
    $error_message = "An error occurred: " . $e->getMessage();
}

$conn->close();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title><?= htmlspecialchars($page_title) ?></title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <!-- STEP 1: Include Flatpickr CSS -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css">
    
    <style>
        /* Scoped styles to avoid affecting other elements */
        .outreach-report-body { font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif; background-color: #f4f6f9; padding: 20px; }
        .outreach-report-container { background-color: #fff; border: 1px solid #dee2e6; border-radius: 8px; padding: 25px; max-width: 1400px; margin: auto; box-shadow: 0 4px 12px rgba(0,0,0,0.05); }
        .outreach-report-header { text-align: center; color: #343a40; margin-bottom: 25px; }
        
        /* Filters Section */
        .outreach-report-filters { display: flex; flex-wrap: wrap; gap: 20px; padding: 20px; background-color: #f8f9fa; border-radius: 6px; margin-bottom: 25px; border: 1px solid #e9ecef;}
        .outreach-report-filters .filter-group { display: flex; flex-direction: column; }
        .outreach-report-filters label { font-weight: 600; font-size: 13px; margin-bottom: 6px; color: #495057; }
        .outreach-report-filters input, .outreach-report-filters select { padding: 8px 12px; border: 1px solid #ced4da; border-radius: 4px; font-size: 14px; transition: border-color 0.2s, box-shadow 0.2s; }
        .outreach-report-filters input:focus, .outreach-report-filters select:focus { border-color: #80bdff; box-shadow: 0 0 0 0.2rem rgba(0,123,255,.25); outline: none; }
        /* Style for flatpickr inputs to show a calendar icon */
        .outreach-flatpickr-input { background-color: #fff; cursor: pointer; }
        
        /* Table Styles */
        .outreach-report-table-wrapper { overflow-x: auto; }
        .outreach-styled-table { width: 100%; border-collapse: collapse; font-size: 14px; }
        .outreach-styled-table th, .outreach-styled-table td { border: 1px solid #ddd; padding: 12px 15px; text-align: left; white-space: nowrap; }
        .outreach-styled-table th { background-color: #f2f2f2; font-weight: bold; color: #495057; }
        .outreach-styled-table tbody tr:nth-of-type(even) { background-color: #f9f9f9; }
        .outreach-styled-table tbody tr:hover { background-color: #e9ecef; }
        .outreach-styled-table tfoot td { font-weight: bold; background-color: #f2f2f2; }
        
        /* Sorting UI */
        .outreach-sortable-header { cursor: pointer; position: relative; }
        .outreach-sortable-header::after { content: '\f0dc'; font-family: 'Font Awesome 6 Free'; font-weight: 900; position: absolute; right: 10px; top: 50%; transform: translateY(-50%); color: #adb5bd; opacity: 0.7; }
        .outreach-sortable-header.sort-asc::after { content: '\f0de'; color: #007bff; }
        .outreach-sortable-header.sort-desc::after { content: '\f0dd'; color: #007bff; }
        
        .outreach-error-message { color: #d93025; font-weight: bold; text-align: center; padding: 20px; }
        .outreach-no-results { color: #6c757d; text-align: center; padding: 20px; }
    </style>
</head>
<body class="outreach-report-body">

    <div class="outreach-report-container">
        <h1 class="outreach-report-header"><?= htmlspecialchars($page_title) ?></h1>

        <!-- STEP 2: Modify the Filter Controls HTML -->
        <div class="outreach-report-filters">
            <div class="filter-group">
                <label for="outreach_report_filter_agent">Agent (User)</label>
                <select id="outreach_report_filter_agent">
                    <option value="all">All Agents</option>
                    <?php foreach ($agent_list as $agent): ?>
                        <option value="<?= htmlspecialchars($agent) ?>"><?= htmlspecialchars($agent) ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
             <div class="filter-group">
                <label for="outreach_report_filter_mode">Mode</label>
                <select id="outreach_report_filter_mode">
                    <option value="all">All Modes</option>
                    <option value="In person">In person</option>
                    <option value="Phone">Phone</option>
                </select>
            </div>
            <div class="filter-group">
                <label for="outreach_report_filter_org">Organisation Name</label>
                <input type="text" id="outreach_report_filter_org" placeholder="Filter by name...">
            </div>
            <div class="filter-group">
                <label for="outreach_report_filter_visit_date">Visit Date Range</label>
                <input type="text" id="outreach_report_filter_visit_date" class="outreach-flatpickr-input" placeholder="Select date range...">
            </div>
            <div class="filter-group">
                <label for="outreach_report_filter_upload_date">Upload Date Range</label>
                <input type="text" id="outreach_report_filter_upload_date" class="outreach-flatpickr-input" placeholder="Select date range...">
            </div>
            <div class="filter-group">
                <label for="outreach_report_filter_next_follow_up_date">Next Follow-up Range</label>
                <input type="text" id="outreach_report_filter_next_follow_up_date" class="outreach-flatpickr-input" placeholder="Select date range...">
            </div>
        </div>

        <?php if ($error_message): ?>
            <p class="outreach-error-message"><?= htmlspecialchars($error_message) ?></p>
        <?php elseif (empty($all_followups)): ?>
            <p class="outreach-no-results">No outreach activity data found in the database.</p>
        <?php else: ?>
            <div class="outreach-report-table-wrapper">
                <table class="outreach-styled-table">
                    <thead>
                        <tr>
                            <th id="outreach_report_header_agent" class="outreach-sortable-header" data-sort-key="agent_user">User (Agent)</th>
                            <th id="outreach_report_header_org" class="outreach-sortable-header" data-sort-key="organization_name">Organisation Name</th>
                            <th id="outreach_report_header_visit" class="outreach-sortable-header" data-sort-key="visit_time">Visit Time</th>
                            <th id="outreach_report_header_mode" class="outreach-sortable-header" data-sort-key="mode">Mode</th>
                            <th id="outreach_report_header_upload" class="outreach-sortable-header" data-sort-key="upload_date">Upload Date</th>
                            <th id="outreach_report_header_next" class="outreach-sortable-header" data-sort-key="next_follow_up_date">Next Follow-up Date</th>
                        </tr>
                    </thead>
                    <tbody id="outreach_report_table_body"></tbody>
                    <tfoot id="outreach_report_table_foot">
                        <tr>
                            <td colspan="5">Total Activities Shown:</td>
                            <td id="outreach_report_total_activities"></td>
                        </tr>
                    </tfoot>
                </table>
            </div>
        <?php endif; ?>
    </div>

    <!-- STEP 1: Include Flatpickr JS (at the end of the body) -->
    <script src="https://cdn.jsdelivr.net/npm/flatpickr"></script>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const outreachReport = {
                allData: <?= json_encode($all_followups) ?>,
                currentSort: { key: 'visit_time', direction: 'desc' },

                // STEP 3: Update the elements object
                elements: {
                    agentFilter: document.getElementById('outreach_report_filter_agent'),
                    visitDateFilter: document.getElementById('outreach_report_filter_visit_date'),
                    uploadDateFilter: document.getElementById('outreach_report_filter_upload_date'),
                    nextFollowUpDateFilter: document.getElementById('outreach_report_filter_next_follow_up_date'),
                    modeFilter: document.getElementById('outreach_report_filter_mode'),
                    orgFilter: document.getElementById('outreach_report_filter_org'),
                    tableBody: document.getElementById('outreach_report_table_body'),
                    totalActivitiesCell: document.getElementById('outreach_report_total_activities'),
                    headers: document.querySelectorAll('.outreach-sortable-header')
                },
                
                // Flatpickr instances
                flatpickrInstances: {},

                init: function() {
                    if (!this.allData || this.allData.length === 0) return;
                    
                    // STEP 3: Initialize Flatpickr instances
                    this.initFlatpickr();
                    
                    this.addEventListeners();
                    this.renderTable();
                },
                
                // NEW Function to set up Flatpickr
                initFlatpickr: function() {
                    const commonOptions = {
                        mode: "range",
                        dateFormat: "Y-m-d",
                        altInput: true, // Creates a more human-readable input field
                        altFormat: "d M Y",
                        onChange: () => this.renderTable(),
                        onReady: function(selectedDates, dateStr, instance) {
                            // Add a clear button to the calendar
                            const clearButton = document.createElement('button');
                            clearButton.className = 'flatpickr-button flatpickr-clear';
                            clearButton.textContent = 'Clear';
                            clearButton.type = 'button';
                            clearButton.addEventListener('click', (e) => {
                                instance.clear();
                                e.stopPropagation();
                            });
                            instance.calendarContainer.appendChild(clearButton);
                        }
                    };

                    this.flatpickrInstances.visit = flatpickr(this.elements.visitDateFilter, commonOptions);
                    this.flatpickrInstances.upload = flatpickr(this.elements.uploadDateFilter, commonOptions);
                    this.flatpickrInstances.nextFollowUp = flatpickr(this.elements.nextFollowUpDateFilter, commonOptions);
                },

                addEventListeners: function() {
                    this.elements.agentFilter.addEventListener('change', () => this.renderTable());
                    this.elements.modeFilter.addEventListener('change', () => this.renderTable());
                    this.elements.orgFilter.addEventListener('input', () => this.renderTable());
                    
                    this.elements.headers.forEach(header => {
                        header.addEventListener('click', (e) => {
                            const sortKey = e.currentTarget.dataset.sortKey;
                            if (this.currentSort.key === sortKey) {
                                this.currentSort.direction = this.currentSort.direction === 'asc' ? 'desc' : 'asc';
                            } else {
                                this.currentSort.key = sortKey;
                                this.currentSort.direction = 'asc';
                            }
                            this.renderTable();
                        });
                    });
                },

                // STEP 3: Update the main render function
                renderTable: function() {
                    const agent = this.elements.agentFilter.value;
                    const mode = this.elements.modeFilter.value;
                    const orgName = this.elements.orgFilter.value.toLowerCase();
                    
                    // Get date ranges from Flatpickr instances
                    const visitDateRange = this.flatpickrInstances.visit.selectedDates;
                    const uploadDateRange = this.flatpickrInstances.upload.selectedDates;
                    const nextFollowUpDateRange = this.flatpickrInstances.nextFollowUp.selectedDates;

                    let filteredData = this.allData.filter(row => {
                        if (agent !== 'all' && row.agent_user !== agent) return false;
                        if (mode !== 'all' && row.mode !== mode) return false;
                        if (orgName && !row.organization_name.toLowerCase().includes(orgName)) return false;
                        
                        // Date range filtering logic
                        if (visitDateRange.length === 2) {
                            const visitDate = row.visit_time ? row.visit_time.split(' ')[0] : null;
                            if (!visitDate || visitDate < this.formatDateForCompare(visitDateRange[0]) || visitDate > this.formatDateForCompare(visitDateRange[1])) return false;
                        }
                        if (uploadDateRange.length === 2) {
                            const uploadDate = row.upload_date ? row.upload_date.split(' ')[0] : null;
                            if (!uploadDate || uploadDate < this.formatDateForCompare(uploadDateRange[0]) || uploadDate > this.formatDateForCompare(uploadDateRange[1])) return false;
                        }
                        if (nextFollowUpDateRange.length === 2) {
                             const nextDate = row.next_follow_up_date ? row.next_follow_up_date.split(' ')[0] : null;
                            if (!nextDate || nextDate < this.formatDateForCompare(nextFollowUpDateRange[0]) || nextDate > this.formatDateForCompare(nextFollowUpDateRange[1])) return false;
                        }
                        
                        return true;
                    });

                    const sortKey = this.currentSort.key;
                    const direction = this.currentSort.direction === 'asc' ? 1 : -1;

                    filteredData.sort((a, b) => {
                        const valA = a[sortKey] || ''; const valB = b[sortKey] || '';
                        if (valA < valB) return -1 * direction; if (valA > valB) return 1 * direction; return 0;
                    });
                    
                    this.elements.tableBody.innerHTML = '';
                    
                    if (filteredData.length === 0) {
                        this.elements.tableBody.innerHTML = '<tr><td colspan="6" class="outreach-no-results">No activities match the current filters.</td></tr>';
                    } else {
                        filteredData.forEach(row => {
                            const tr = document.createElement('tr');
                            tr.innerHTML = `
                                <td>${this.escapeHtml(row.agent_user)}</td>
                                <td>${this.escapeHtml(row.organization_name)}</td>
                                <td>${this.formatDisplayDate(row.visit_time, true)}</td>
                                <td>${this.escapeHtml(row.mode)}</td>
                                <td>${this.formatDisplayDate(row.upload_date, true)}</td>
                                <td>${this.formatDisplayDate(row.next_follow_up_date, false)}</td>
                            `;
                            this.elements.tableBody.appendChild(tr);
                        });
                    }

                    this.elements.totalActivitiesCell.textContent = filteredData.length;
                    
                    this.elements.headers.forEach(header => {
                        header.classList.remove('sort-asc', 'sort-desc');
                        if (header.dataset.sortKey === this.currentSort.key) {
                            header.classList.add(`sort-${this.currentSort.direction}`);
                        }
                    });
                },

                formatDisplayDate: function(dateString, includeTime) {
                    if (!dateString) return 'N/A';
                    try {
                        const date = new Date(dateString);
                        const options = { year: 'numeric', month: 'short', day: 'numeric' };
                        if (includeTime) {
                            options.hour = 'numeric'; options.minute = '2-digit'; options.hour12 = true;
                        }
                        return date.toLocaleDateString('en-GB', options);
                    } catch (e) { return dateString; }
                },

                // NEW Helper to format dates for raw comparison (YYYY-MM-DD)
                formatDateForCompare: function(date) {
                    return date.toISOString().split('T')[0];
                },
                
                escapeHtml: function(str) {
                    if (str === null || str === undefined) return '';
                    const p = document.createElement("p"); p.textContent = String(str); return p.innerHTML;
                }
            };
            outreachReport.init();
        });
    </script>
</body>
</html>