<?php
// qbo_ab_test_all_entities.php
// The final, comprehensive A/B test for all entities synced by the cron job.
// Proves that the "Plan B" sync method is correct for Invoices, Customers, and Items.

set_time_limit(1800);
ini_set('memory_limit', '512M');

require_once __DIR__ . '/qbo_bootstrap.php';

// --- Configuration ---
const FORCED_PAGES_TO_FETCH = 20;
const RESULTS_PER_PAGE = 10; // Using smaller batch size as requested.
$testStartDate = (new DateTime())->sub(new DateInterval('P30D'))->format('Y-m-d');
$lookbackDate = (new DateTime())->sub(new DateInterval('P45D'))->format('Y-m-d\TH:i:sP');

function write_header($title) {
    echo "\n\n======================================================================\n";
    echo "  $title\n";
    echo "======================================================================\n";
}

// Generic function to run the entire A/B test for a given entity
function testEntity($dataService, $entityName, $dateColumn = 'TxnDate') {
    write_header("A/B TEST FOR ENTITY: {$entityName}");

    // --- METHOD A (CONTROL): FORCE FETCH ---
    echo "--- Running Method A (Control): Force Fetch ---\n";
    $control_ids = [];
    for ($i = 0; $i < FORCED_PAGES_TO_FETCH; $i++) {
        $startPosition = 1 + ($i * RESULTS_PER_PAGE);
        // For Customer/Item, we filter by create time as they don't have a TxnDate
        $dateFilter = ($entityName === 'Invoice')
            ? "WHERE {$dateColumn} >= '{$GLOBALS['testStartDate']}'"
            : "WHERE MetaData.CreateTime >= '{$GLOBALS['testStartDate']}'";
            
        $query = "SELECT Id FROM {$entityName} {$dateFilter} STARTPOSITION {$startPosition} MAXRESULTS " . RESULTS_PER_PAGE;
        
        echo "  Fetching page " . ($i + 1) . "/" . FORCED_PAGES_TO_FETCH . "... ";
        $entities = $dataService->Query($query);
        
        if (empty($entities)) { echo "Done.\n"; break; }
        
        $count = count($entities);
        echo "Found {$count}.\n";
        foreach ($entities as $entity) { $control_ids[] = $entity->Id; }
    }
    $control_ids = array_unique($control_ids);

    // --- METHOD B (CHALLENGER): LAST MODIFIED TIME ---
    echo "--- Running Method B (Challenger): Last Modified Time ---\n";
    $challenger_ids = [];
    $i = 1;
    while (true) {
        $query = "SELECT * FROM {$entityName} WHERE MetaData.LastUpdatedTime > '{$GLOBALS['lookbackDate']}' ORDERBY MetaData.LastUpdatedTime ASC STARTPOSITION {$i} MAXRESULTS " . RESULTS_PER_PAGE;
        echo "  Fetching batch (Position {$i})... ";
        $entities = $dataService->Query($query);

        if (empty($entities)) { echo "Done.\n"; break; }
        
        $count = count($entities);
        echo "Found {$count}.\n";
        foreach ($entities as $entity) {
            // Filter by the appropriate date column to match the control group
            $entityDate = ($entityName === 'Invoice') ? $entity->TxnDate : $entity->MetaData->CreateTime;
            if ($entityDate >= $GLOBALS['testStartDate']) {
                $challenger_ids[] = $entity->Id;
            }
        }
        $i += $count;
    }
    $challenger_ids = array_unique($challenger_ids);

    // --- ANALYSIS ---
    echo "--- Analysis for {$entityName} ---\n";
    sort($control_ids);
    sort($challenger_ids);

    echo "Total Unique Found by CONTROL:   " . count($control_ids) . "\n";
    echo "Total Unique Found by CHALLENGER: " . count($challenger_ids) . "\n";

    $missed_by_challenger = array_diff($control_ids, $challenger_ids);
    $missed_by_control = array_diff($challenger_ids, $control_ids);

    if (empty($missed_by_challenger) && empty($missed_by_control)) {
        echo "\n--------------------------------------------------------\n";
        echo "  ✅  TEST PASSED for {$entityName} - DATASETS ARE IDENTICAL!  ✅\n";
        echo "--------------------------------------------------------\n";
    } else {
        echo "\n--------------------------------------------------------\n";
        echo "  ❌  TEST FAILED for {$entityName} - DATASETS ARE DIFFERENT!  ❌\n";
        echo "--------------------------------------------------------\n";
        if (!empty($missed_by_challenger)) {
            echo "MISSED by CHALLENGER: \n"; print_r($missed_by_challenger);
        }
        if (!empty($missed_by_control)) {
            echo "MISSED by CONTROL: \n"; print_r($missed_by_control);
        }
    }
}

// --- Main Execution ---
$dataService = $qboUtil->getDataService();

testEntity($dataService, 'Invoice');
testEntity($dataService, 'Customer');
testEntity($dataService, 'Item');

write_header("A/B TESTING COMPLETE");
?>